/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.authorization.policy;

import java.net.URL;
import java.security.AccessControlContext;
import java.security.CodeSource;
import java.security.DomainCombiner;
import java.security.Principal;
import java.security.ProtectionDomain;
import java.security.cert.Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import javax.security.auth.Subject;

import net.sf.jguard.core.authorization.domaincombiners.RestrictDomainCombiner;
import net.sf.jguard.core.authorization.domaincombiners.StackSubjectDomainCombiner;
import net.sf.jguard.core.principals.RolePrincipal;

/**
 * 
 * utility class for authorization work related to {@link AccessControlContext} and {@link ProtectionDomain}.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * @author <a href="mailto:vberetti@users.sourceforge.net">Vincent Beretti</a>
 * @see java.security.AccessControlContext
 * @see java.security.ProtectionDomain
 */
public class AccessControlContextUtils {

	/**
	  * return the convenient {@link AccessControlContext} corresponding to the principal.
	  * the returned AccessControlContext is bound to a {@link RestrictDomainCOmbiner}.
	  * @param principal RolePrincipal used to restrict execution code rights
	  * @return object embedding used to restrict permissions
	  */
	  public static AccessControlContext getRestrictedAccessControlContext(Principal principal){
	      ProtectionDomain pd = ProtectionDomainUtils.getEmptyProtectionDomain(principal);
	      DomainCombiner restrictDomainCombiner = new RestrictDomainCombiner();
	      AccessControlContext acc = new AccessControlContext(new ProtectionDomain[]{pd});
	      AccessControlContext acc2 = new AccessControlContext(acc,restrictDomainCombiner);
		  return acc2;
	  }

	/**
	   * gets an <code>AccessControlContext</code> containing a single <code>ProtectionDomain</code>
	   * with an <code>null</code> <code>CodeSource</code>, an empty array of <code>Certificates</code>,
	   * the current <code>Thread</code> <code>ClassLoader</code>, and the subject principals.
	   * @param subject
	   * @return the generated AccessControlContext
	   */
	  public static AccessControlContext getSubjectOnlyAccessControlContext(Subject subject){
		  ProtectionDomain pd = new ProtectionDomain(new CodeSource((URL)null,(Certificate[])null),null,Thread.currentThread().getContextClassLoader(),(Principal[])subject.getPrincipals().toArray(new Principal[subject.getPrincipals().size()]));
		  ProtectionDomain[] pds = new ProtectionDomain[1];
		  pds[0] = pd;
		  AccessControlContext acc = new AccessControlContext(pds);
		  return acc;
	  }

	public static AccessControlContext  getStackSubjectAccessControlContext(Subject subject){
		  ProtectionDomain[] arrayPd = new ProtectionDomain[0];
		  AccessControlContext acc = new AccessControlContext(arrayPd);
		  DomainCombiner dc = new StackSubjectDomainCombiner(subject);
		  AccessControlContext acc2 = new AccessControlContext(acc,dc);
		  return acc2;
	  }

	/**
	   * return the convenient {@link AccessControlContext} containing the collection of Principal
	   * but no permissions and a fake [@link CodeSource}.
	   * the current Policy will further bound permissions to these principals.
	   * @param principals RolePrincipal used to restrict execution code rights
	   * @return object used to restrict permissions
	   */
	  public static AccessControlContext getAccessControlContext(Collection principals){
		  Iterator itPrincipals = principals.iterator();
		  Collection protectionDomains = new ArrayList();
		  while(itPrincipals.hasNext()){
			  RolePrincipal principal = (RolePrincipal)itPrincipals.next();
			  protectionDomains.add(ProtectionDomainUtils.getEmptyProtectionDomain(principal));
		  }
		  AccessControlContext acc = new AccessControlContext((ProtectionDomain[]) protectionDomains.toArray(new ProtectionDomain[protectionDomains.size()]));
		  return acc;
	  }

}
