/*
 jGuard is a security framework based on top of jaas (java authentication and authorization security).
 it is written for web applications, to resolve simply, access control problems.
 version $Name$
 http://sourceforge.net/projects/jguard/

 Copyright (C) 2004  Charles GAY

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


 jGuard project home page:
 http://sourceforge.net/projects/jguard/

 */
package net.sf.jguard.core.authorization.permissions;

import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.AccessController;
import java.security.Guard;
import java.security.Permission;
import java.util.ArrayList;
import java.util.Collection;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * It represents the right to execute one or more actions (or methods for HTTP) 
 * on one or more URLs via one or more protocols identified by their schemes.
 * this permission, <strong>only</strong> implies URLPermission.
 * @see java.lang.IllegalArgumentException which wrap the
 * @see java.net.URISyntaxException thrown if the URI is not correct.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 *
 */
public final class URLPermission extends java.security.BasicPermission implements Serializable, Cloneable, Comparable {

	/** Logger for this class */
	private static final Logger logger = Logger.getLogger(URLPermission.class.getName());
	
	//HTTP methods 
	public static final String DELETE ="DELETE";
	public static final String GET ="GET";
	public static final String HEAD ="HEAD";
	public static final String OPTIONS ="OPTIONS";
	public static final String POST ="POST";
	public static final String PUT ="PUT";
	public static final String TRACE ="TRACE";
	public static final String ANY ="ANY";

	//protocoles
	public static final String HTTP="http";
	public static final String HTTPS="https";
	
	
	
	/**
	 * serial version id.
	 */
	private static final long serialVersionUID = 3257283643243574067L;

	private Pattern pattern;

	/**
	 * regexp to display (include /* instead of /.*)
	 */
	private String prettyPattern;

	private URI uri;

	/**
	 * unique permission's name.
	 */
	private String name;

	/**
	 * explain permission
	 */
	private String description = "";

	private URLParameterCollection parameters;
	
	private Collection methods = new ArrayList();

	/**
	 * protocol (http, https...). default value is 'http'
	 */
	private String scheme= URLPermission.ANY;

	/**
	 * actions
	 */
	private StringBuffer actions = new StringBuffer();

	/**
	 * Creates a new instance of URLPermission.
	 *
	 * @param n
	 *            permission's name
	 */
	public URLPermission(String n) {
		super(n);
		this.name = n;
		try {
			uri = new URI("");
		} catch (URISyntaxException e) {
			throw new IllegalArgumentException(e.getMessage());
		}
		parameters = new URLParameterCollection();
	}

	/**
	 * Creates a new instance of URLPermission.
	 *
	 * @param name
	 *            permission name
	 * @param actions  permission's actions splitted by a "," :
	 *  regexp,scheme(optional),description(optional),http methods(optional)
	 *  http methods and schemes (Http or https) are automatically recognized, after the regexp.
	 * @throws IllegalArgumentException
	 *             which wraps a
	 * @see URISyntaxException
	 */
	public URLPermission(String name, String actions) {
		super(name);
		this.name = name;
		String[] actionsArray = actions.split(",");

		if (actionsArray.length > 4)
			throw new IllegalArgumentException(" 'actions' argument can contain a maximum of three elements separated by ',' ");

		try {
			setURI(actionsArray[0]);
		} catch (URISyntaxException e) {
			throw new IllegalArgumentException(e.getMessage());
		}

		
		for(int i =1;i<actionsArray.length;i++){
			
			if(URLPermission.HTTPS.equalsIgnoreCase(actionsArray[i])||
			   URLPermission.HTTP.equalsIgnoreCase(actionsArray[i])){
				this.scheme  = actionsArray[i];
				continue;
			}else if(URLPermission.DELETE.equalsIgnoreCase(actionsArray[i])
					||URLPermission.GET.equalsIgnoreCase(actionsArray[i])
					||URLPermission.HEAD.equalsIgnoreCase(actionsArray[i])
					||URLPermission.OPTIONS.equalsIgnoreCase(actionsArray[i])
					||URLPermission.POST.equalsIgnoreCase(actionsArray[i])
					||URLPermission.PUT.equalsIgnoreCase(actionsArray[i])
					||URLPermission.TRACE.equalsIgnoreCase(actionsArray[i])){
				methods.add(actionsArray[i]);
				continue;
				
			}else{
				this.description = actionsArray[i];
				continue;
			}
				
		}

		//no scheme specified implies any scheme
		if(scheme==null){
			scheme = URLPermission.ANY;
		}
		
		//no methods specified implies all methods
		if(methods.size()==0){
			methods.add(URLPermission.ANY);
		}
		
		// Mount the string for actions

		// regexp output form
		this.actions.append(this.prettyPattern);

		// scheme
		this.actions.append(',');
		this.actions.append(this.scheme);

		// description
		if (this.description.length() > 0) {
			this.actions.append(',');
			this.actions.append(this.description);
		}

	}

	/**
	 * 'prettyPattern' is the regexp included in the URI : uri with a 'star'(*) operator, which can be evaluated.
	 *
	 * @param pPattern
	 * @throws URISyntaxException 
	 */
	private void setURI(String pPattern) throws URISyntaxException {

		// build the regexp pattern
		String regexp = pPattern;
		regexp = buildRegexpFromString(getPathFromURIString(regexp));

		pattern = Pattern.compile(regexp);
		if (logger.isLoggable(Level.FINEST)) {
			logger.log(Level.FINEST, "regexp=" + regexp);
		}
		String uriWithoutRegexp = removeRegexpFromURI(pPattern);
		this.uri = new URI(uriWithoutRegexp);

		if (logger.isLoggable(Level.FINEST)) {
			logger.finest("uri=" + uri);
		}

		prettyPattern = pPattern;

		parameters = URLParameterCollection.getURLParameters(getQueryFromURIString(pPattern));
	}

	/**
	 * replace '*'character (not followed by '*' character, or if it's the last '*') by '' and we replace '**' by '*'.
	 *
	 * @param uriPath
	 * @return URI escaped
	 */
	static public String removeRegexpFromURI(String uriPath) {
		// we replace '*'character (not followed by '*' character, or if it's
		// the last '*') by ''
		// => '*' in the regexp is replaced
		uriPath = uriPath.replaceAll("\\*(?!\\*)", "");
		if (logger.isLoggable(Level.FINEST)) {
			// logger.log(Level.FINEST,"uriPath="+uriPath);
		}
		// we replace '**' by '*' => we convert the escaped(double) '*' in the
		// simple '*'
		uriPath = uriPath.replaceAll("\\*{2}", "\\*");

		// --------------- added by VBE
		// replace '$' by UTF-8 symbol for '$'
		uriPath = uriPath.replaceAll("\\$\\{", "%24%7B");
		// replace '}' by UTF-8 symbol for '}'
		uriPath = uriPath.replaceAll("\\}", "%7D");
		// --------------- end added by VBE

		return uriPath;
	}

	public static String getPathFromURIString(String uriString) {
		String uriPath = uriString;
		int position = uriString.indexOf("?");
		if (position != -1) {
			uriPath = uriString.substring(0, position);
		}
		return uriPath;
	}

	public static String getQueryFromURIString(String uriString) {
		String uriQuery = "";
		int position = uriString.indexOf("?");
		if (position != -1) {
			uriQuery = uriString.substring(position + 1, uriString.length());
		}
		return uriQuery;
	}

	/**
	 * convenient method to escape regexp special characters, and only use the '*' characters for building the regexp Pattern.
	 *
	 * @param regexp
	 * @return escaped regexp candidate
	 */
	public static String buildRegexpFromString(String regexp) {

		// replace '\' by '\\'
		regexp = regexp.replaceAll("\\\\", "\\\\\\\\");
		// replace '**' by '\*\*'
		regexp = regexp.replaceAll("\\*\\*", "\\\\*\\\\*");
		// replace '?' by '\\?'
		regexp = regexp.replaceAll("\\?", "\\\\\\\\?");
		// replace '+' by '\\+'
		regexp = regexp.replaceAll("\\+", "\\\\\\\\+");
		// replace '{' by '\\{'
//		regexp = regexp.replaceAll("\\{", "\\\\\\\\{");
		// replace '}' by '\\}'
//		regexp = regexp.replaceAll("\\}", "\\\\\\\\}");
		// replace '[' by '\\['
		regexp = regexp.replaceAll("\\[", "\\\\\\\\[");
		// replace ']' by '\\]'
		regexp = regexp.replaceAll("\\[", "\\\\\\\\]");
		// replace '^' by '\\^'
		regexp = regexp.replaceAll("\\^", "\\\\\\\\^");
		// replace '$' by '\\$'
//		regexp = regexp.replaceAll("\\$", "\\\\\\\\$");

		// replace '&' by '\\&'
		regexp = regexp.replaceAll("\\&", "\\\\\\\\&");

		// replace '*' by '\.*'
		regexp = regexp.replaceAll("\\*", "\\.\\*");
		return regexp;
	}

	/**
	 * Determines whether or not to allow access to the guarded object object.
	 *  this method comes from the {@link Guard} interface.
	 *
	 * @param perm  Permission to check
	 */
	public void checkGuard(Object perm) {
		Permission p = (Permission) perm;
		AccessController.checkPermission(p);
	}

	/**
	 * override the java.lang.Object 's <i>clone</i> method.
	 *
	 * @return new URLPermission with the <strong>same Domain</strong>.
	 */
	public Object clone() throws CloneNotSupportedException {

		URLPermission permission = null;
		permission = new URLPermission(this.name, this.getActions());
		return permission;

	}

	/**
	 * @param obj
	 * @return true if equals, false otherwise
	 *
	 */
	public boolean equals(Object obj) {
		if ((obj instanceof URLPermission) && ((URLPermission) obj).getName().equals(this.getName())) {
			// we compare two URLPermission with the same name
			URLPermission tempPerm = (URLPermission) obj;
			
			String[] tempActions = tempPerm.getActions().split(",");
			URI tempUri = null;
			try {
				tempUri = new URI(removeRegexpFromURI(tempActions[0]));
			} catch (URISyntaxException e) {
				logger.log(Level.SEVERE, " URI syntax error: " + removeRegexpFromURI(tempActions[0]),e);
			}

			if(!tempPerm.getScheme().equals(this.scheme)){
				return false;
			}
			
			if(!tempPerm.getMethods().equals(this.methods)){
				return false;
			}
			
			// we should compare paths of these URLPermissions and parameters
			// but parameter order
			// doesn't mind
			if (uri.getPath().equals(tempUri.getPath())){
				if(uri.getQuery() == null && tempUri.getQuery() == null){
					return true;
				}else if(uri.getQuery() == null || tempUri.getQuery() == null){
					return false;
				}else if(uri.getQuery().equals(tempUri.getQuery())) {
					return true;
				}
			}
			return false;
		}
		return false;
	}

	/**
	 * return actions in a String splitted by comma.
	 *
	 * @return permitted actions.
	 */
	public String getActions() {
		return actions.toString();
	}

	/**
	 * methode used to accelerate the comparation process: useful when hashcode return different int.
	 *
	 * @return hashcode
	 */
	public int hashCode() {
		return name.hashCode();
	}

	/**
	 * verify if this permission implies another URLPermission.
	 *
	 * @param permission
	 * @return true if implies, false otherwise
	 */
	public boolean implies(java.security.Permission permission) {
		URLPermission urlpTemp = null;
		if (!(permission instanceof URLPermission)) {
			if (logger.isLoggable(Level.FINEST)) {
				logger.log(Level.FINEST, " permission is not an URLPermission. type = " + permission.getClass().getName());
			}
			return false;

		}
		

		urlpTemp = (URLPermission) permission;
		
		if(this.equals(permission)){
			return true;
		}
		
		//test ations
		String urlpTempActions = urlpTemp.getActions();
		if( urlpTempActions == null ||"".equals(urlpTempActions)){
			if( actions == null ||"".equals(actions.toString())){
				return true;	
			}
			return false;
		}
		
		//test scheme
		if(!this.scheme.equals(URLPermission.ANY)&& !this.scheme.equals(urlpTemp.getScheme())){
			return false;
		}
		
		//test methods
		if(!this.methods.contains(URLPermission.ANY)){
			Collection httpMethods = new ArrayList(urlpTemp.getMethods());
			httpMethods.retainAll(this.methods);
			if(httpMethods.size()==0){
			return false;
			}
		}
		
		boolean b = impliesURI(urlpTemp.getURI());
		
		// test the parameters
		// only if the uri is right
		if (!b) {
			return false;
		}
		
		b = impliesParameters(getQueryFromURIString(urlpTemp.getURI()));
		
		if (logger.isLoggable(Level.FINEST)) {
			logger.finest("access decision =" + b);
		}
		return b;

	}

	private boolean impliesURI(String uri) {
		String regexp = getPathFromURIString(uri);
		Matcher m = pattern.matcher(regexp);
		if (logger.isLoggable(Level.FINEST)) {
			logger.log(Level.FINEST, "pattern used to check access =" + pattern.pattern());
			logger.log(Level.FINEST, " path to be checked =" + regexp);
		}
		boolean b = m.matches();
		if (logger.isLoggable(Level.FINEST)) {
			logger.log(Level.FINEST, "access decision =" + b);
		}
		m.reset();
		return b;
	}

	/**
	 * look at the provided parameters by the user permission.
	 *
	 * @param queryFromUserPermission
	 * @return
	 */
	private boolean impliesParameters(String queryFromUserPermission) {

		if("".equals(queryFromUserPermission)){
			queryFromUserPermission = null;
		}

		// if the permission hasn't got any parameters
		// => the permission implies any parameters
		if (queryFromUserPermission != null && !parameters.isEmpty()) {
			String[] params = queryFromUserPermission.split("&");

			// iterate over required parameters keys/values
			for (int a = 0; a < params.length; a++) {
				String[] keyAndValue = params[a].split("=");
				URLParameter urlparam = new URLParameter();
				urlparam.setKey(keyAndValue[0]);
				String[] values = new String[1];
				if (keyAndValue.length != 1){
					values[0] = keyAndValue[1];
				}else{
					values[0] = "";
				}
				urlparam.setValue(values);
				// if the evaluated permissionList does not contain the right
				// URLParameter
				if (!parameters.implies(urlparam)) {
					return false;
				}

			}
		} else if (parameters.isEmpty() && queryFromUserPermission != null) {
			return true;
		} else if (parameters.isEmpty() && queryFromUserPermission == null) {
			return true;
		} else if (!parameters.isEmpty() && queryFromUserPermission == null) {
			return false;
		}

		// we have iterate over the permission collection
		// and have found each URLParameter in the evaluated permission
		return true;
	}

	/**
	 * return an enmpy JGPermissionCollection.
	 *
	 * @return empty JGPermissionCollection
	 */
	public java.security.PermissionCollection newPermissionCollection() {
		return new JGPositivePermissionCollection();
	}

	/**
	 * return a String representation of the permission.
	 *
	 * @return String
	 */
	public String toString() {

		StringBuffer sb = new StringBuffer();
		sb.append(" name: " + this.name);
		sb.append("\n scheme: " + this.scheme);
		sb.append("\n parameters: " + this.parameters.toString());
		sb.append("\n pattern: " + this.pattern);
		sb.append("\n uri: " + this.uri);
		sb.append("\n description: " + this.description);
		sb.append("\n");

		return sb.toString();
	}

	/**
	 * method used to compare two objects. this method is used in Collection to <strong>order</strong> items, and MUST be
	 * consistent with the <i>equals</i> method (eache method should return 0/true in the same cases).
	 *
	 * @param o
	 *            object to compare
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 * @return 0 if both objects are equals, &lt;0 if 0 is lesser than the URLPermission, &gt;0 if 0 is greater than the
	 *         URLPermission
	 */
	public int compareTo(Object o) {

		URLPermission perm = (URLPermission) o;
		if (this.equals(perm)) {
			return 0;
		}
		return this.name.compareTo(perm.getName());
	}

	public final String getURI() {
		return prettyPattern;
	}

	public Collection getMethods() {
		return methods;
	}

	public String getScheme() {
		return scheme;
	}

}