/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.authorization.permissions;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.security.BasicPermission;
import java.security.Permission;
import java.security.PermissionCollection;
import java.security.Permissions;

import java.security.Principal;
import java.security.ProtectionDomain;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheException;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;
import net.sf.jguard.core.principals.RolePrincipal;
import net.sf.jguard.core.principals.UserPrincipal;

import org.apache.commons.jexl.Expression;
import org.apache.commons.jexl.ExpressionFactory;
import org.apache.commons.jexl.JexlContext;
import org.apache.commons.jexl.JexlHelper;



/**
 * java.security.Permission related utility class.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * @author <a href="mailto:vberetti@users.sourceforge.net">Vincent Beretti</a>
 * @author <a href="mailto:tandilero@users.sourceforge.net">Maximiliano Batelli</a>
 */
public class PermissionUtils {

    private static final Logger logger = Logger.getLogger(PermissionUtils.class.getName());

    private static CacheManager manager;
	private static Cache unresolvedPermToNeededExpressions;
	private static Cache unresolvedPermAndValuesToResolvedPerm;
    private static boolean cachesEnabled;
    private static Pattern JEXL_PATTERN = Pattern.compile("(\\$\\{[^\\}]+\\})");

    /**
     * instantiate a java.security.Permission subclass.
     * @param className class name
     * @param name permission name
     * @param actions actions name split by comma ','
     * @return a java.security.Permission subclass, or a java.security.BasicPermission subclass
     * (which inherit java.security.Permission)
     * @throws ClassNotFoundException 
     */
    public static Permission getPermission(String className, String name,String actions) throws ClassNotFoundException{
    	
         Class clazz = null;
        try {
            clazz = Thread.currentThread().getContextClassLoader().loadClass(className);
        } catch (ClassNotFoundException e1) {
            logger.log(Level.SEVERE," class "+className+" is not found please check your classPath \n and the permission set in the Datasource \n(either database or JGuardPrincipalsPermissions.xml file) ",e1);
            throw e1;
        }
         Class[] permArgsBasicPermClass = {String.class,String.class};
         Class[] permArgsPermClass = {String.class};
         Object[] objBasicArray = {name,actions};
         Permission newPerm = null;
         try {
                 //check if className inherit from the Abstract BasicPermission class which
                 // has got a two string argument constructor to speed up the lookup
                 if(clazz.isAssignableFrom(BasicPermission.class)){
                      newPerm = (Permission) clazz.getConstructor(permArgsBasicPermClass).newInstance(objBasicArray);
                      return newPerm;
                 }
         
                 Object[] objArray = {name};

                 Constructor[] constructors = clazz.getConstructors();
                 boolean constructorWithActions = false;
                 for(int i = 0;i<constructors.length;i++){
                     Constructor tempConstructor = constructors[i];
                     Class[] classes = tempConstructor.getParameterTypes();
                     if(classes.length==2 && classes[0].equals(String.class)&& classes[1].equals(String.class)){
                         constructorWithActions  = true;
                         break;
                     }
                 }
        
                // a class which does not inherit from BasicPermission but has got a two string arguments constructor
                if(constructorWithActions == true){
                     newPerm = (Permission) clazz.getConstructor(permArgsBasicPermClass).newInstance(objBasicArray);
                }else{
                    //Permission subclass which has got a constructor with name argument
                     newPerm = (Permission) clazz.getConstructor(permArgsPermClass).newInstance(objArray);
                }
        } catch (IllegalArgumentException e) {
            logger.log(Level.SEVERE," illegal argument ",e);
        } catch (SecurityException e) {
            logger.log(Level.SEVERE,"className="+className);
            logger.log(Level.SEVERE,"name="+name);
            logger.log(Level.SEVERE,"actions="+actions);
            logger.log(Level.SEVERE," you don't have right to instantiate a permission ",e);
        } catch (InstantiationException e) {
            logger.log(Level.SEVERE,"className="+className);
            logger.log(Level.SEVERE,"name="+name);
            logger.log(Level.SEVERE,"actions="+actions);
            logger.log(Level.SEVERE," you cannot instantiate a permission ",e);
        } catch (IllegalAccessException e) {
            logger.log(Level.SEVERE,"className="+className);
            logger.log(Level.SEVERE,"name="+name);
            logger.log(Level.SEVERE,"actions="+actions);
            logger.log(Level.SEVERE,e.getMessage(),e);
        } catch (InvocationTargetException e) {
            logger.log(Level.SEVERE,"className="+className);
            logger.log(Level.SEVERE,"name="+name);
            logger.log(Level.SEVERE,"actions="+actions);
            logger.log(Level.SEVERE,e.getMessage(),e);
        } catch (NoSuchMethodException e) {
        	logger.log(Level.SEVERE,"method not found =",e);
        }
        return newPerm;
    }

    /**
     * Evaluate jexlExpression using UserPrincipal as context.<br>
     * and return <strong>true</strong> if expression is valid,
     *  <strong>false</strong> otherwise.
     * @param jexlExpression
     * @param userPrincipal
     * @return boolean
     */
    private static boolean evaluateDefinition(String jexlExpression, UserPrincipal userPrincipal) {
        final String PRIVATE_CREDENTIALS = "subject.privateCredentials";
        final String PUBLIC_CREDENTIALS = "subject.publicCredentials";
        final String ROLES = "subject.roles";
        final String ORGANIZATION = "subject.organization";
    	if(jexlExpression == null)
    		return false;
    	if("true".equalsIgnoreCase(jexlExpression))
    		return true;
    	if("false".equalsIgnoreCase(jexlExpression))
    		return false;
    	if(jexlExpression != null && userPrincipal == null) {
			logger.warning("evaluateDefinition() no UserPrincipal defined, can not use regex definition");
    	}
    	
            jexlExpression = jexlExpression.substring(2, jexlExpression.length()-1);
            JexlContext jexlContext = JexlHelper.createContext();
            jexlContext.getVars().put(ORGANIZATION, userPrincipal.getOrganization());
            jexlContext.getVars().put(ROLES, userPrincipal.getRoles());
            jexlContext.getVars().put(PUBLIC_CREDENTIALS, userPrincipal.getPublicCredentials());
            jexlContext.getVars().put(PRIVATE_CREDENTIALS, userPrincipal.getPrivateCredentials());

            Object resolvedExpression = null;
            try {
                    Expression expression = ExpressionFactory.createExpression(jexlExpression);
                    resolvedExpression = expression.evaluate(jexlContext);
            } catch (Exception e) {
                    logger.warning("Failed to evaluate : " + jexlExpression);
            }

            if (resolvedExpression == null || !(resolvedExpression instanceof Boolean)){
                    logger.warning("Subject does not have the required credentials to resolve the role activation : "+ jexlExpression);
                    return false;
            } else {
                    Boolean val = (Boolean)resolvedExpression;
                    return val.booleanValue();
            }
    }

    /**
     * Evaluate principal definition attr and active attr.<br>
     * To resolve definition attr, this method uses a particular Principal (UserPrincipal)
     * set to the user during authentication. If this principal is not present and the definition attr != null, the
     * definition attr is not evaluated and the function returns false.
     * definition attr take precedence against active attr, so 
     * if definition evaluate to false but active is true, then evaluatePrincipal return false 
     * @param ppal RolePrincipal to evaluate
     * @param userPrincipal UserPrincipal used to evaluate ppal parameter
     * @return boolean
     */
    public static boolean evaluatePrincipal(RolePrincipal ppal, UserPrincipal userPrincipal) {
    	if(!evaluateDefinition(ppal.getDefinition(), userPrincipal)) {
			if (logger.isLoggable(Level.FINEST)) {
    			logger.finest("evaluatePrincipal() -  user's principal definition attr evaluates to false="+ ppal.getLocalName());
    		}
			return false;
    	} else if(!ppal.isActive()) {
			if (logger.isLoggable(Level.FINEST)) {
    			logger.finest("evaluatePrincipal() -  user's principal active attr is false="+ ppal.getLocalName());
    		}
			return false;
    	} else
    		return true;

    }
    
    /**
     * Resolve permission collection containing regular expressions.<br>
     * To resolve the permissions, this method uses a particular Principal (UserPrincipal)
     * set to the user during authentication. If this principal is not present, the
     * permission collection given in parameters is returned with no modifications. If
     * the UserPrincipal is present but does not contain the required data to resolved the regex,
     * the permission is removed from the permission collection.
     * @param protectionDomain
     * @param pc
     * @return
     */
	public static PermissionCollection evaluatePermissionCollection(ProtectionDomain protectionDomain, PermissionCollection pc){
        final String PRIVATE_CREDENTIALS = "subject.privateCredentials";
        final String PUBLIC_CREDENTIALS = "subject.publicCredentials";
        final String SUBJECT_ROLES = "subject.roles";
        
		// resolve regular expressions in permissions
		Principal[] ppals = protectionDomain.getPrincipals();
		boolean hasJexlPrincipal = false;
		int i = 0;
		
		//we are looking for UserPrincipal to resolve regexp with JEXL
		while (!hasJexlPrincipal && i < ppals.length){
			hasJexlPrincipal = ppals[i] instanceof UserPrincipal;
			i++;
		}
		if (!hasJexlPrincipal){
			logger.warning("no UserPrincipal defined, can not use regex permissions");
			return pc;
		}else {
			PermissionCollection resolvedPc = new JGPositivePermissionCollection();

			UserPrincipal subjectPrincipal = (UserPrincipal)ppals[i-1];
			JexlContext jc = JexlHelper.createContext();
			Map vars= jc.getVars();
			vars.put(SUBJECT_ROLES, subjectPrincipal.getRoles());
			vars.put(PUBLIC_CREDENTIALS, subjectPrincipal.getPublicCredentials());
			vars.put(PRIVATE_CREDENTIALS, subjectPrincipal.getPrivateCredentials());
			
			//TODO CGA add time-based permissions with DurationDecorator class
			
			Enumeration permissionsEnum = pc.elements();

			Map subjectResolvedExpressions = new HashMap();
			// stores every already resolved expressions inside this method i.e. for a subject principal
			while (permissionsEnum.hasMoreElements()){
				Permission permission = (Permission)permissionsEnum.nextElement();
				logger.finest("Resolving permission = " + permission);
				PermissionCollection pcFromPermission = resolvePermission(permission, subjectResolvedExpressions, jc);
				Enumeration enumPermissions = pcFromPermission.elements();
				while(enumPermissions.hasMoreElements()){
					Permission p = (Permission) enumPermissions.nextElement();
					resolvedPc.add(p);
				}
			}

			return resolvedPc;
		}
	}

	private static HashSet createKey(Permission unresolvedPermission, Map values){

		HashSet key = new HashSet();
		key.add(unresolvedPermission);
		key.add(values);

		return key;
	}

	/**
	 * return all the permissions which match the regexp expression present in the 
	 * permission passed as a parameter.
	 * @param permission to resolve
	 * @param subjectResolvedExpressions Map containing JEXL expression as <strong>key</strong>, and Resolved Permission as <strong>value</strong>
	 * @param jexlContext JEXL context containing variables used to resolve permissions.
	 * @return resolved PermissionCollection
	 */
	private static PermissionCollection resolvePermission (Permission permission, Map subjectResolvedExpressions, JexlContext jexlContext){

		PermissionCollection resolvedPermissions = new JGPositivePermissionCollection();

		// try to get the resolved permissions from the cache
		if (cachesEnabled){
			try {
				// check in cache if unresolved permission -> needed expressions exists
				Element expressionsCacheEntry = unresolvedPermToNeededExpressions.get(permission);
				if (expressionsCacheEntry != null){

					Set neededExpressions =(Set) expressionsCacheEntry.getValue();

					if (neededExpressions.isEmpty()){
						// no need to resolve this permission
						resolvedPermissions.add(permission);
						logger.finest("get permission from cache with no resolution needed");
						return resolvedPermissions;
					}

					Iterator itExpressions = neededExpressions.iterator();
					Map permissionResolvedExpressions = new HashMap();
					boolean hasNull = false;
					while (itExpressions.hasNext()){
						String jexlExpression = (String) itExpressions.next();
						Object resolvedExpression = null;

						if (subjectResolvedExpressions.containsKey(jexlExpression)){
							resolvedExpression = subjectResolvedExpressions.get(jexlExpression);
							permissionResolvedExpressions.put(jexlExpression, resolvedExpression);
						}else {
							try {   
								Expression expression = ExpressionFactory.createExpression(jexlExpression);
                                                                //resolution work made by JEXL is done here
								resolvedExpression = expression.evaluate(jexlContext);
								subjectResolvedExpressions.put(jexlExpression, resolvedExpression);
								permissionResolvedExpressions.put(jexlExpression, resolvedExpression);
							} catch (Exception e) {
								logger.warning("Failed to evaluate : " + jexlExpression);
							}
						}

						if (resolvedExpression == null || (resolvedExpression instanceof List && ((List)resolvedExpression).isEmpty())){
							hasNull = true;
							break;
						}

					}

					if (hasNull){
						logger.warning("Subject does not have the required credentials to resolve the permission : "+ permission);
						//skip this unresolvable permission
						resolvedPermissions.add(permission);
						return resolvedPermissions;
					}

					// check in cache if (needed values + unresolvedPermission) -> resolved permission exists
					HashSet key = createKey(permission, permissionResolvedExpressions);
					Element permissionCacheEntry = unresolvedPermAndValuesToResolvedPerm.get(key);

					if (permissionCacheEntry != null){
						PermissionCollection permissionsFromCache = (PermissionCollection) permissionCacheEntry.getValue();
						logger.finest("get resolved permission from cache");
						Enumeration enumeration = permissionsFromCache.elements();
						while(enumeration.hasMoreElements()){
							Permission permissionFromCache = (Permission) enumeration.nextElement();
							resolvedPermissions.add(permissionFromCache);
						}
						return resolvedPermissions;
					}
				}
			} catch (CacheException e) {
				logger.log(Level.WARNING, "Failed using caches : " + e.getMessage());
			}
		}

		// if permission is not yet resolved continue
		// resolution will be fast because jexlExpression -> value
		// has already been resolved and stored in resolvedValues

		// resolution is combinative so one unresolved permission
		// may imply n resolved permissions
		List unresolvedPermissions = new ArrayList();
		unresolvedPermissions.add(permission);
		Map resolvedExpressionsByPermission = new HashMap();

		while (!unresolvedPermissions.isEmpty()) {

			Permission unresolvedPermission = (Permission) unresolvedPermissions.remove(0);

			String name = unresolvedPermission.getName();
			Set partiallyResolvedNames = resolvePartiallyExpression(name, JEXL_PATTERN, jexlContext, resolvedExpressionsByPermission, subjectResolvedExpressions);
			if(partiallyResolvedNames == null){
				// unresolvable permission
				return new JGPositivePermissionCollection();
			}

			boolean matchesInName = (partiallyResolvedNames.size() != 1 || !partiallyResolvedNames.contains(name));
			if (matchesInName) {
				Iterator itNames = partiallyResolvedNames.iterator();
				while (itNames.hasNext()) {
					String resolvedName = (String) itNames.next();
					Permission partiallyResolvedPermission;
					try {
						partiallyResolvedPermission = PermissionUtils.getPermission(permission.getClass().getName(), resolvedName, unresolvedPermission.getActions());
					} catch (ClassNotFoundException e) {
						logger.warning(e.getMessage());
						continue;
					}
					unresolvedPermissions.add(partiallyResolvedPermission);
				}
				continue;
			}

			String actions = unresolvedPermission.getActions();
                        if(actions==null){
                            actions="";
                        }
			String[] actionsArray = actions.split(",");
			String action = actionsArray[0];
			Set partiallyResolvedActions = resolvePartiallyExpression(action, JEXL_PATTERN, jexlContext, resolvedExpressionsByPermission, subjectResolvedExpressions);
			if(partiallyResolvedActions == null){
				// unresolvable permission
				return new JGPositivePermissionCollection();
			}

			boolean matchesInActions = (partiallyResolvedActions.size() != 1 || !partiallyResolvedActions.contains(action));
			if (matchesInActions) {
				Iterator itActions = partiallyResolvedActions.iterator();
				while (itActions.hasNext()) {
					String resolvedAction = (String) itActions.next();
					Permission partiallyResolvedPermission;
					try {
						partiallyResolvedPermission = PermissionUtils.getPermission(permission.getClass().getName(), unresolvedPermission.getName(), resolvedAction);
					} catch (ClassNotFoundException e) {
						logger.warning(e.getMessage());
						continue;
					}
					unresolvedPermissions.add(partiallyResolvedPermission);
				}
				continue;
			}

			// if this code is reached, there is no match in name and actions
			// the permission is resolved
			resolvedPermissions.add(unresolvedPermission);
		}

		if (cachesEnabled){
			try {
				// store permissions needed expressions in cache
				List unresolvedKeys = unresolvedPermToNeededExpressions.getKeys();
				if (!unresolvedKeys.contains(permission)){

					HashSet permissionNeededExpressions = new HashSet(resolvedExpressionsByPermission.keySet());
					unresolvedPermToNeededExpressions.put(new Element(permission, permissionNeededExpressions));
				}
			} catch (CacheException e) {
				logger.log(Level.WARNING, "Failed using caches : " + e.getMessage());
			}

			// store mapping (values + unresolved permission ) -> resolved permission in cache
			Element cacheEntry = new Element(createKey(permission, resolvedExpressionsByPermission), resolvedPermissions);
			unresolvedPermAndValuesToResolvedPerm.put(cacheEntry);
			logger.finest("add resolved permissions to cache");
		}

		return resolvedPermissions;
	}


	/**
	 * /**
	 * resolves first occurence of jexl expression. The other expressions remain unresolved
	 * @param expression
	 * @param pattern
	 * @param jexlContext
	 * @param resolvedExpressionsByPermission
	 * @param subjectResolvedExpressions
	 * @return
	 */
	private static Set resolvePartiallyExpression (String expression, Pattern pattern, JexlContext jexlContext, Map resolvedExpressionsByPermission, Map subjectResolvedExpressions){

		boolean hasMatches = false;
		boolean hasNull = false;

		Set resolvedExpressionsSet = new HashSet();

		Matcher matcher = pattern.matcher(expression);
		if (matcher.find()) {
			hasMatches = true;
			String matchedExpression = matcher.group();

			String jexlExpression = matchedExpression.substring (2, matchedExpression.length()-1);
			Object resolvedExpression = null;

			if (subjectResolvedExpressions.containsKey(jexlExpression)) {
				resolvedExpression = (Set) subjectResolvedExpressions.get(jexlExpression);
			} else {
				try {
					Expression expr = ExpressionFactory.createExpression(jexlExpression);
					resolvedExpression = expr.evaluate(jexlContext);
					subjectResolvedExpressions.put(jexlExpression, resolvedExpression);

				} catch (Exception e) {
					logger.warning("Failed to resolve expression : " + jexlExpression);
				}
			}

			if ( !(resolvedExpressionsByPermission.containsKey(jexlExpression))){
				resolvedExpressionsByPermission.put(jexlExpression, resolvedExpression);
			}

			if(resolvedExpression == null){
				// expression can not be resolved
				hasNull = true;
			}else if(resolvedExpression instanceof Set) {
				Iterator it = ((Set)resolvedExpression).iterator();
				while(it.hasNext()){
					StringBuffer builder = new StringBuffer(expression);
					builder.replace(matcher.start(),matcher.end(),(String)it.next());
					resolvedExpressionsSet.add(builder.toString());
				}
			}else if(resolvedExpression instanceof String){
				StringBuffer builder = new StringBuffer(expression);
				builder.replace(matcher.start(),matcher.end(),(String)resolvedExpression);
				resolvedExpressionsSet.add(builder.toString());
			}
		}

		if (!hasMatches){
			// no jexl expression in part, return original part
			resolvedExpressionsSet.add(expression);
		}
		if (hasNull){
			// can not be resolved
			return null;
		}

		return resolvedExpressionsSet;
	}

	public static void createCaches() throws CacheException{
		// gets ehcache.xml as a resource in the classpath
		if(unresolvedPermToNeededExpressions == null ||
				unresolvedPermAndValuesToResolvedPerm == null ){
			logger.info("Creating caches for permissions evaluations");
			manager = CacheManager.create();
			unresolvedPermToNeededExpressions = manager.getCache("unresolvedPermToNeededExpressions");
			unresolvedPermAndValuesToResolvedPerm = manager.getCache("unresolvedPermAndValuesToResolvedPerm");

			if(unresolvedPermToNeededExpressions == null || unresolvedPermAndValuesToResolvedPerm == null){
				logger.warning("Failed to create caches for permissions evaluations, use non-caching evaluation");
				setCachesEnabled(false);
			}
		}
	}

	public static boolean isCachesEnabled() {
		return cachesEnabled;
	}

	public static void setCachesEnabled(boolean cachesEnabled) {
		PermissionUtils.cachesEnabled = cachesEnabled;
	}
        
         /**
         * create an heterogenous PermissionCollection( formerly, a Permission<b>s<b/> instance
         * which will contains the content of the two PermissionCollections.
         */
        public  static Permissions mergePermissionCollections(PermissionCollection perm1,PermissionCollection perm2){
            Permissions result = new Permissions();
            Enumeration enumPerm1 = perm1.elements();
            while(enumPerm1.hasMoreElements()){
                Permission perm = (Permission)enumPerm1.nextElement();
                result.add(perm);
            }
            Enumeration enumPerm2 = perm2.elements();
            while(enumPerm2.hasMoreElements()){
                Permission perm = (Permission)enumPerm2.nextElement();
                result.add(perm);
            }
            return result;
        }
}