/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.authorization.permissions;

import java.security.Permission;
import java.security.PermissionCollection;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.logging.Logger;



/**
 * contains similar permissions.
 * this class contains similar <strong>java.security.Permission</strong> instances,
 *  with the same type.
 * it is a "technical" container in opposite to Domain,which is a "functional" container.
 * Classes extending this abstract class must implements implies method from PermissionCollection.
 * @author <a href="mailto:diabolo512@users.sourceforge.net ">Charles Gay</a>
 *
 */
public abstract class JGPermissionCollection extends PermissionCollection {

	/**
	 * serial version id.
	 */
	private static final long serialVersionUID = 3834030277143377201L;
	/** Logger for this class */
	private static final Logger logger = Logger.getLogger(JGPermissionCollection.class.getName());

	protected Set permissions;

	/**
     * default constructor.
     *
     */
    public JGPermissionCollection() {

        permissions = new HashSet();
    }

    /**
     * constructor.
     * @param coll
     */
    public JGPermissionCollection(Collection coll) {

        permissions = new HashSet(coll);
    }
	/**
	 * add a permission to the set.
	 * @see java.security.PermissionCollection#add(java.security.Permission)
	 */
	public void add(Permission permission) {
            if(permission!=null){
		permissions.add(permission);
            }

	}

    /**
     * add permissions to the set.
     * @param permissionSet
     * @see java.security.PermissionCollection#add(java.security.Permission)
     */
    public void addAll(Set permissionSet) {
        if(permissionSet!=null){
            permissions.addAll(permissionSet);
        }

    }
    
    public void addAll(PermissionCollection pcColl){
    	Enumeration en = pcColl.elements();
        while(en.hasMoreElements()){
        	permissions.add((Permission)en.nextElement());
        }
    }

	/**
	 * return all the permissions.
	 * @see java.security.PermissionCollection#elements()
	 */
	public Enumeration elements() {
        return Collections.enumeration(permissions);

	}

   



   /**
    * return the corresponding permission.
    * @param permissionName
    * @return permission
 * @throws NoSuchPermissionException
    */
    public Permission getPermission (String permissionName) throws NoSuchPermissionException{
        Permission permission;
        Iterator it = permissions.iterator();
        while(it.hasNext()){
            permission = (Permission)it.next();
            if(permission.getName().equals(permissionName)){
                return permission;
            }

        }
        logger.warning("permission "+permissionName+" not found in JGPermissionCollection#getPermission!!!");
        logger.warning("permissions="+permissions);
        throw new NoSuchPermissionException("permission "+permissionName+" not found in JGPermissionCollection#getPermission");

    }

    /**
     * remove permission from Permission's collection.
     * @param permission
     */
    public void removePermission(Permission permission){
        if(permission!= null){
            permissions.remove(permission);
        }
    }

    /**
     * remove permission from Permission's collection.
     * @param permission
     */
    public void removePermissions(PermissionCollection permColl){
    	Enumeration permissionsEnum = permColl.elements();
    	while(permissionsEnum.hasMoreElements()){
    		permissions.remove((Permission)permissionsEnum.nextElement());
    	}
    }
    
    /**
     * remove permission from Permission's collection.
     * @param permission
     */
    public void clear(){
    	permissions.clear();
    }
    
    public String toString(){
    	StringBuffer sb = new StringBuffer();
    	Iterator permissionsIterator = this.permissions.iterator();
    	while(permissionsIterator.hasNext()){
    		Permission permission = (Permission)permissionsIterator.next();
    		sb.append(permission.toString());
    		sb.append("\n");
    	}
    	return sb.toString();
    }

    /**
     *
     * @return permissions number owned by this JgPermissionCollection.
     */
    public int size(){
    	return permissions.size();
    }

	/**
	 * @return Returns the permissions.
	 */
	public Set getPermissions() {
	    return permissions;
	}

	/**
	 *
	 * @param permission
	 * @return
	 */
	public boolean containsPermission(Permission permission) {
	     return permissions.contains(permission);
	}

	/**
	 * @param perms The permissions to set.
	 */
	public void setPermissions(Set perms) {
	    this.permissions = perms;
	}
	
}
