/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.authentication.manager;



import java.security.Principal;
import java.util.Collection;
import java.util.Set;

import javax.security.auth.Subject;

import net.sf.jguard.core.authentication.AuthenticationException;
import net.sf.jguard.core.authentication.credentials.JGuardCredential;
import net.sf.jguard.core.organization.Organization;
import net.sf.jguard.core.provisioning.OrganizationTemplate;
import net.sf.jguard.core.provisioning.RegistrationException;
import net.sf.jguard.core.provisioning.SubjectTemplate;

/**
 *
 * this interface provide the ability to operate on the system which stores user profiles.
 * it can be a database, or an ldap server, or anything else....
 * it concerns <strong>administrators</strong> guys.
 * there is one AuthenticationManager per webapp.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * @author <a href="mailto:tandilero@users.sourceforge.net">Maximiliano Batelli</a>
 *
 */
public interface AuthenticationManager {

     public Organization createOrganization(OrganizationTemplate organizationTemplate)throws RegistrationException;
     public void deleteOrganization(Organization organisation);
     
     /**
      * return the <i>default</i> Organization identified by the id <b>system</b>.
      */
     public Organization getDefaultOrganization();
     public Set getOrganizations()throws AuthenticationException;
     public void updateOrganization(String organizationIdentityCredential, Organization organisation)throws AuthenticationException;
     
     /**
      * @param unique organizationId to find
      * @return Organization found or <b>null</b> otherwise.
      */
     public Organization findOrganization(String organizationId);
   
    public String getCredentialId();
    
    public String getCredentialPassword();

    /**
     * add principals in the list to the persistance storage.
     * create a Principal in the backend, <strong>only</strong> if it is not already present.
     * @param role
     * @throws AuthenticationException
     */
    public  void createPrincipal(Principal role)  throws AuthenticationException ;
    
    /**
     * @param user to be controlled
     * @param subjectTemplate is the validator
     */
    public Subject createUser(SubjectTemplate user,Organization organization) throws RegistrationException;
     /**
     * @param user to be controlled
     * @param subjectTemplate is the validator
     */ 
    public Subject createUser(Subject user,Organization organization) throws AuthenticationException ;
    /**
     * retrieve role from the webapp.
     * @param name
     * @return roleInterface
     * @throws AuthenticationException
     */
    public Principal getLocalPrincipal(String name)throws AuthenticationException;


    /**
     * get the principals defined in the repository for all the applications.
     * @return role's list.
     * @throws AuthenticationException
     */
    public Set getAllPrincipalsSet() throws AuthenticationException;

    /**
     * get the principals defined in the repository <strong>only</strong> for this application.
     * @return role's list.
     *
     */
    public Set getLocalPrincipals();

    /**
     * retrieve users which are in conformance with all these credentials.
     * @param credentials
     * @return users
     * @throws AuthenticationException
     */
    public Set findUsers(Collection privateCredentials,Collection publicCredentials)throws AuthenticationException;

    /**
     * checks if a user with the same username already exists.
     * @param user
     * @return result. true if a user with the same name exists, false otherwise.
     * @throws AuthenticationException
     */
    public boolean userAlreadyExists(Subject user) throws AuthenticationException;


	/**
	 * role wether or not exists in the webapp.
	 * @param role to check
	 * @return result
	 * @throws AuthenticationException
	 */
	public boolean hasPrincipal(Principal role)throws AuthenticationException;

	/**
	 * role wether or not exists in the webapp.
	 * @param role to check
	 * @return result
	 * @throws AuthenticationException
	 */
	public boolean hasPrincipal(String role)throws AuthenticationException;

	/**
	 * update user's informations.
	 * @param identityCred
	 * @param user
	 * @throws AuthenticationException
	 */
	public void updateUser(JGuardCredential identityCred,Subject user)throws AuthenticationException;

	/**
	 * remove user.
	 * @param user
	 * @throws AuthenticationException
	 */
	public void deleteUser(Subject user)throws AuthenticationException;

	/**
	 * return the OrganizationTemplate.
	 * @param name
	 * @return SubjectTemplate
	 * @throws AuthenticationException
	 */
    public OrganizationTemplate getOrganizationTemplate() throws AuthenticationException;

    public void setOrganizationTemplate(OrganizationTemplate organizationTemplate)throws AuthenticationException;
    
    
    public Set getUsers() throws AuthenticationException;

    public boolean isEmpty();

    /**
     * change principal's name.
     * @param oldPrincipalName
     * @param principal
     * @throws AuthenticationException
     */
	public void updatePrincipal(String oldPrincipalName, Principal principal) throws AuthenticationException;


	/**
	 * delete this principal and its references in users.
	 * @param principal
	 * @return false i delete fails, true if it succeed
	 * @throws AuthenticationException
	 */
	public boolean deletePrincipal(Principal principal)throws AuthenticationException ;


    /**
     * Clone a Principal with a random name
     * @param roleName Principal name to clone
     * @return cloned Principal with a different name: roleName + Random integer betweeen 0 and 99999
     * @throws AuthenticationException
     */
	public Principal clonePrincipal(String roleName) throws AuthenticationException;

	/**
     * Clone a Principal. If Principal is instance of RolePrincipal makes a call to the clone method leting the clone task to RolePrincipal
     * @param roleName Principal name to clone
     * @param cloneName Principal cloned name
     * @return cloned Principal with the given cloneName
     * @throws AuthenticationException
     */
    public Principal clonePrincipal(String roleName, String cloneName) throws AuthenticationException;
    /**
     * change 'active' property on the specified role for a user.
     * this change cannot be done on 'guest' user, or if it remains only one 'active=true' role.
     * @param subject
     * @param roleName
     * @param applicationName
     * @param active
     * @throws AuthenticationException
     */
    public void setActiveOnRolePrincipal(Subject subject,String roleName,String applicationName,boolean active) throws AuthenticationException;
    
    public Subject findUser(String login);
    
    public void updateRoleDefinition(Subject subject, String role,String applicationName,String definition) throws AuthenticationException;
    
    
    public void importAuthenticationManager(AuthenticationManager authManager);
    
    /**
     * Persists a user logon attempt.
     * @param login user unique ID
     * @param hostAdress IP Adress
     * @param hostname DNS Name bound to the IP adress
     * @succeeded boolean authentication result
     */
    public void persistUserAccessAttempt(String login, String hostAddress, String hostName, boolean succeeded,long timeStamp) throws AuthenticationException;
    
    /**
     * define the name of the <strong>current</strong> application which holds this 
     * AuthenticationManager.
     */
    public String getApplicationName();
}
