/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name:  $
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.authentication.configuration;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import java.util.logging.Logger;
import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.AppConfigurationEntry;
import javax.security.auth.login.Configuration;
import javax.security.auth.login.LoginException;
import javax.security.auth.login.AppConfigurationEntry.LoginModuleControlFlag;
import javax.security.auth.spi.LoginModule;


	/**
	 * LoginContext only 'local' to this application.
	 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
	 * @since 1.0
	 */
	public class LocalLoginContext {
		private static final Logger logger = Logger.getLogger(LocalLoginContext.class.getName());
		private static final String OTHER = "other";
		private List appEntriesList = null;
		private CallbackHandler cbHandler = null;
		private Subject subject = null;
		private Map flags = null;
		private boolean loginSucceed = true;
		private boolean subjectNotProvided;
		private List loginModules = null;
		
		/**
		 * constructor which mimics {@link LoginContext} constructor.
		 * @param name
		 * @param cbHandler
		 * @throws LoginException
		 */
		public LocalLoginContext(String name,CallbackHandler cbHandler) throws LoginException {
			appEntriesList = getAppConfigurationEntry(Configuration.getConfiguration(),name);
			this.cbHandler =cbHandler; 
			subject = new Subject();
			flags = new HashMap();
			subjectNotProvided = true;
		}
		
		/**
		 * constructor which mimics {@link LoginContext} constructor.
		 * @param name
		 * @param cbHandler
		 * @throws LoginException
		 */
		public LocalLoginContext(String name,CallbackHandler cbHandler,Configuration configuration) throws LoginException {
			appEntriesList = getAppConfigurationEntry(configuration,name);
			this.cbHandler =cbHandler; 
			subject = new Subject();
			flags = new HashMap();
			subjectNotProvided = true;
		}
		/**
		 * constructor which mimics {@link LoginContext} constructor.
		 * @param name
		 * @param subject
		 * @param cbHandler
		 * @throws LoginException
		 */
		public LocalLoginContext(String name,Subject subject,CallbackHandler cbHandler) throws LoginException {
			appEntriesList = getAppConfigurationEntry(Configuration.getConfiguration(),name);
			this.cbHandler =cbHandler;
			this.subject = subject;
			flags = new HashMap();
		}
		
		/**
		 * constructor which permits to have for java SE 4 this constructor prsent only on java SE 5. 
		 * @param name
		 * @param subject
		 * @param cbHandler
		 * @param configuration
		 * @throws LoginException
		 */
		public LocalLoginContext(String name,Subject subject,CallbackHandler cbHandler,Configuration configuration) throws LoginException {
			appEntriesList = getAppConfigurationEntry(configuration,name);
			this.cbHandler =cbHandler; 
			this.subject = subject;
			flags = new HashMap();
		}
		
		
		private List getAppConfigurationEntry(Configuration config,String name){
			AppConfigurationEntry[] appEntries = null;
			appEntries = config.getAppConfigurationEntry(name);
			if (appEntries==null){
				appEntries = config.getAppConfigurationEntry(OTHER);
			}
			return Arrays.asList(appEntries);
		}
		
		/**
		 * perform the authentication in a <strong>local</strong> manner,
		 *  i.e, not bound to the JVM's Configuration,
		 * and if successful, associate Principals and
		 * credentials with the Authenticated Subject.
		 * @throws LoginException
		 */
		public void login() throws LoginException{
			
			loginModules = initializeLoginModules(appEntriesList,subject,cbHandler);
			
			//login phase
			Iterator itLoginModules = loginModules.iterator();
			//first overall authentication exception
			LoginException exception = null;
			while(itLoginModules.hasNext()){
				LoginModule module = (LoginModule)itLoginModules.next();
				LoginModuleControlFlag flag = (LoginModuleControlFlag) flags.get(module);
				
				//loginModule Flag check
				if((!LoginModuleControlFlag.OPTIONAL.equals(flag))
					&&(!LoginModuleControlFlag.REQUIRED.equals(flag))
					&&(!LoginModuleControlFlag.REQUISITE.equals(flag))
					&&(!LoginModuleControlFlag.SUFFICIENT.equals(flag))){
						
					logger.severe(" loginModule="+module.getClass()+" has got an invalid flag="+flag);
					logger.severe(" this loginModule is skipped in the authentication process ");
					continue;
				}
					
				//login phase
				try {
					boolean loginModuleSucceed = module.login();
					//loginModule should be ignored => we skip it and continue to the next one
					//according to the Loginmodule javadoc 
					if(!loginModuleSucceed){
						logger.finest(" loginModule "+module.getClass()+" in 'login' phase is ignored  ");
						continue;
					}
					
					logger.finest(" loginModule "+module.getClass()+" in 'login' phase succeed ");
					
					//login succeed
					if(LoginModuleControlFlag.REQUIRED.equals(flag)){
						continue;
					}else if(LoginModuleControlFlag.REQUISITE.equals(flag)){
						continue;
					}else if(LoginModuleControlFlag.SUFFICIENT.equals(flag)){
						break;
					}else if(LoginModuleControlFlag.OPTIONAL.equals(flag)){
						continue;
					}
				} catch (LoginException e) {
					//we store only the first exception in the overall authentication process
					if(exception==null){
						exception = e;	
					}
					
					logger.finest(" loginModule "+module.getClass()+" in 'login' phase failed ");
					//login fails
					logger.info(" authentication fails "+e.getMessage());
					if(LoginModuleControlFlag.REQUIRED.equals(flag)){
						loginSucceed = false;
						continue;
					}else if(LoginModuleControlFlag.REQUISITE.equals(flag)){
						loginSucceed = false;
						break;
					}else if(LoginModuleControlFlag.SUFFICIENT.equals(flag)){
						continue;
					}else if(LoginModuleControlFlag.OPTIONAL.equals(flag)){
						continue;
					}
				}	
				
			}
			Iterator itLoginModules2 = loginModules.iterator();
			
			if(loginSucceed){
				//commit phase
				while(itLoginModules2.hasNext()){
					LoginModule module = (LoginModule)itLoginModules2.next();
					try {
						boolean moduleCommitSucceed = module.commit();
						if(moduleCommitSucceed){
							logger.finest(" loginModule "+module.getClass()+" in 'commit' phase succeeed");
						}else{
							logger.finest(" loginModule "+module.getClass()+" in 'commit' phase is ignored ");
						}
					} catch (LoginException e) {
						logger.finest(" loginModule "+module.getClass()+" in 'commit' phase failed  ");
						abort(loginModules,e);
						throw e;
					}
				}
				
			}else{
				abort(loginModules,exception);
			}
			
		}

		private void abort(List loginModules,LoginException exception) throws LoginException {
			Iterator itLoginModules = loginModules.iterator();
			//abort phase
			while(itLoginModules.hasNext()){
				LoginModule module = (LoginModule)itLoginModules.next();
				try {
					boolean moduleAbortSucceed = module.abort();
					if(moduleAbortSucceed){
						logger.finest(" loginModule "+module.getClass()+" in 'abort' phase succeeed");
					}else{
						logger.finest(" loginModule "+module.getClass()+" in 'abort' phase is ignored ");
					}
				} catch (LoginException e) {
					logger.finest(" loginModule "+module.getClass()+" in 'abort' phase failed ");
					logger.warning(e.getMessage());
					throw exception;
				}
			}
			//we throw the initial exception which causes abort phase
			throw exception;
		}

		private List initializeLoginModules(List appConfigurationEntries,Subject subject ,CallbackHandler cbHandler) {
			Map sharedState = new HashMap();
			List loginModules = new ArrayList();
			Iterator itAppEntries = appConfigurationEntries.iterator();
			//we initialize loginmodules
			while(itAppEntries.hasNext()){
				AppConfigurationEntry entry = (AppConfigurationEntry)itAppEntries.next();
				LoginModuleControlFlag flag = entry.getControlFlag();
				String loginModuleName =entry.getLoginModuleName();
				Map options = entry.getOptions();
				Class loginModuleClass = null;
				LoginModule module = null;
				//grab loginModule Class
				try {
					loginModuleClass = (Class)Thread.currentThread().getContextClassLoader().loadClass(loginModuleName);
				} catch (ClassNotFoundException e) {
					logger.severe(" loginModule Class "+loginModuleName+" not found ");
					throw new RuntimeException("loginModule "+loginModuleName+" is not found "+e.getMessage());
				}
				//instantiate it
				try {
					module = (LoginModule) loginModuleClass.newInstance();
				} catch (InstantiationException e) {
					logger.severe(" loginModule Class "+loginModuleName+" cannot be instantiated  ");
					throw new RuntimeException(e.getMessage());
				} catch (IllegalAccessException e) {
					logger.severe(" loginModule Class "+loginModuleName+" cannot be accessed  ");
					throw new RuntimeException(e.getMessage());
				}
				
				//initialize it
				module.initialize(subject,cbHandler,sharedState,options);
				flags.put(module,flag);
				loginModules.add(module);
			}
			
			return loginModules;
		}
		
		public void logout() throws LoginException{
			LoginException exception = null;
                        if(loginModules==null){
                            return;
                        }
			Iterator itLoginModules = loginModules.iterator();
			while(itLoginModules.hasNext()){
				LoginModule module =(LoginModule)itLoginModules.next();
				try {
					boolean moduleLogoutSucceed = module.logout();
					if(moduleLogoutSucceed){
						logger.finest(" loginModule "+module.getClass()+" in 'logout' phase succeeed");
					}else{
						logger.finest(" loginModule "+module.getClass()+" in 'logout' phase is ignored ");
					}
				} catch (LoginException e) {
					logger.finest(" loginModule "+module.getClass()+" in 'logout' phase failed ");
					logger.severe(e.getMessage());
					if (exception == null){
						exception = e;
					}
				}
			}
			
			//we throw the first exception raised by loginModules
			//but we try to logout before all the loginModules
			if(exception != null){
				throw exception;
			}
		}
		
		public Subject getSubject(){
			if(!loginSucceed && subjectNotProvided){
				return null;
			}
			return subject;
		}
	}
