/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.

http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.authentication.configuration;


import java.security.AccessController;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.security.auth.AuthPermission;
import javax.security.auth.login.AppConfigurationEntry;
import javax.security.auth.login.Configuration;

/**
 * extends the <a href="http://java.sun.com/j2se/1.4.2/docs/api/javax/security/auth/login/Configuration.html">
 * Configuration</a>
 * this class is used to define the authentication stack scheme per application.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 *
 */
public final class JGuardConfiguration extends Configuration {

    private Collection internalConfigs = null;
    private Map appConfigurations = null;
    private static Logger logger = Logger.getLogger(JGuardConfiguration.class.getName());
    /**
     * constructor.
     */
    public JGuardConfiguration() {
        super();
        logger.finest("#####  JGuardConfiguration  #####");

        internalConfigs = new ArrayList();
        appConfigurations = new HashMap();

    }

    /**
     * reload the Configuration.
     * @see javax.security.auth.login.Configuration#refresh()
     */
    public void refresh() {
        AccessController.checkPermission(new AuthPermission("refreshLoginConfiguration"));

        if(internalConfigs.size()>0){
            Iterator itConfigs = internalConfigs.iterator();
            while(itConfigs.hasNext()){
                Configuration tempConfig = (Configuration) itConfigs.next();
                tempConfig.refresh();
            }
        }

    }

    /**
     * retrieve the AppConfigurationEntry array for the corresponding application's name.
     * @param applicationName
     * @return array of AppConfigurationEntry
     * @see javax.security.auth.login.Configuration#getAppConfigurationEntry(java.lang.String)
     */
    public AppConfigurationEntry[] getAppConfigurationEntry(
            String applicationName) {
        Collection appInternalEntries = new ArrayList();
        Iterator itConfigs = internalConfigs.iterator();
        while(itConfigs.hasNext()){
        	Configuration tempConfig = (Configuration) itConfigs.next();
        	if (tempConfig == null){
        		itConfigs.remove();
        		logger.log(Level.WARNING,"the default Configuration implementation has been removed from the JGuardConfiguration which imported it");
        	} else if( tempConfig.getAppConfigurationEntry(applicationName)!= null){
        		appInternalEntries.addAll(Arrays.asList(tempConfig.getAppConfigurationEntry(applicationName)));
        	}
        }

        List jGuardAppConfigEntries = (List)appConfigurations.get(applicationName);

       if(jGuardAppConfigEntries!=null){
           appInternalEntries.addAll(jGuardAppConfigEntries);
       }
        if(appInternalEntries.size()>0){
            return (AppConfigurationEntry[]) appInternalEntries.toArray(new AppConfigurationEntry[appInternalEntries.size()]);
        }

        return null;

    }


    /**
     * add AppconfigurationEntries for a specified application.
     * @param applicationName
     * @param entries
     * @see javax.security.auth.login.Configuration#getAppConfigurationEntry(java.lang.String)
     */
    public  void addConfigEntriesForApplication(String applicationName,List entries) {
    	if(entries == null || entries.size() == 0){
    		logger.log(Level.WARNING," entries provided are null or empty ");
	        return;
	    }

	    List applicationEntries = (List) appConfigurations.get(applicationName);
	    if(applicationEntries==null){
	    	//this application is not yet configured
	        appConfigurations.put(applicationName,entries);
	    }

	    //we don't add other appConfigurationEntries if the application name already exists
	    //because when webapp stops and start (but not app server), configuration for the application
	    // will have twice loginmodules
    }

    /**
     * add the same AppconfigurationEntries like an already configured application.
     * @param applicationName name of the application to configure
     * @param applicationTemplateName name of the application
     * which will be the template to configure the first one.
     * @see javax.security.auth.login.Configuration#getAppConfigurationEntry(java.lang.String)
     */
    public  void addConfigEntriesLikeApplication(String applicationName,String applicationTemplateName) {
        List applicationEntries = (List) appConfigurations.get(applicationTemplateName);
        if(applicationEntries== null){
        	logger.log(Level.SEVERE," there is no applications registered with your applicationName and password ");
        	return;
        }
        appConfigurations.put(applicationName,applicationEntries);
    }

    /**
     * add the same AppconfigurationEntries like an already configured application.
     * @param applicationName
     * @param password
     * @param applicationTemplateName
     */
    public  void addConfigEntriesLikeApplication(String applicationName,String password,String applicationTemplateName) {
    	StringBuffer newApplicationName = new StringBuffer(applicationName)
        .append('#').append(password);
    	addConfigEntriesLikeApplication(newApplicationName.toString(),applicationTemplateName);
    }

    /**
     * include Configuration information.
     * @param configuration
     */
    protected void includeConfiguration(Configuration configuration){
    	//we do not include a jGuardConfiguration to prevent infinite loop
    	if(!configuration.getClass().getName().equals(JGuardConfiguration.class.getName())
    	&&!internalConfigs.contains(configuration)){
         internalConfigs.add(configuration);
    	}
    }


	/**
	 * register the application with a name and a password
	 * add AppconfigurationEntries for a specified application.
	 * @param applicationName
	 * @param password
	 * @param entries
	 * @see javax.security.auth.login.Configuration#getAppConfigurationEntry(java.lang.String)
	 */
	public  void addConfigEntriesForApplication(String applicationName,String password,List entries) {
		StringBuffer newApplicationName = new StringBuffer(applicationName)
        .append('#').append(password);
		addConfigEntriesForApplication(newApplicationName.toString(),entries);
	}


}
