/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/
package net.sf.jguard.core.authentication;

import java.io.IOException;
import java.util.Locale;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.LanguageCallback;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.login.Configuration;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;

import net.sf.jguard.core.authentication.callbacks.InetAddressCallback;
import net.sf.jguard.core.authentication.configuration.LocalLoginContext;
import net.sf.jguard.core.authentication.manager.AuthenticationManager;
import net.sf.jguard.core.authentication.manager.AuthenticationManagerFactory;
import net.sf.jguard.core.principals.UserPrincipal;
import net.sf.jguard.core.util.ThrowableUtils;


/**
 * Authentication wrapper around the {@link Subject}, which permits to infer
 *  on the Subject's LifeCycle (login, logout). 
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 */
public class AuthenticationUtils {

	private LoginContext loginContext = null;
	private LocalLoginContext localLoginContext = null;
	private Subject subject = null;
	private boolean loggedOut = false;
	private boolean local = false;
        private static final Logger logger = Logger.getLogger(AuthenticationUtils.class.getName());
	private Configuration configuration = null;
	
	public AuthenticationUtils(){
		super();
	}
	
	
	public AuthenticationUtils(Configuration config){
		super();
		configuration = config;
		local= true;
	}
	
	/**
	 * authenticate user against the application's configuration.
	 * @param applicationName
	 * @param cbh 
	 * @throws LoginException raised if authentication failed
	 */
	public void login( String applicationName, CallbackHandler cbh) throws LoginException {
            
                InetAddressCallback inetCbk = new InetAddressCallback();
                NameCallback nameCallback = new NameCallback(" ");
                boolean authenticationSucceed = false;
                Locale locale = Locale.getDefault();
                LanguageCallback languageCallback = new LanguageCallback();
                try{
                       
                        //we define our needs by the callbacks array
                        Callback[] callbacks = new Callback[3];
                        callbacks[0]= inetCbk;
                        callbacks[1]= nameCallback;
                        callbacks[2] = languageCallback;
                        try {
                            cbh.handle(callbacks);
                            locale = languageCallback.getLocale();
                        } catch (UnsupportedCallbackException ex) {
                            throw new LoginException(ex.getMessage());
                        } catch (IOException ex) {
                            throw new LoginException(ex.getMessage());
                        }
                        
                        if(local){
                            localLoginContext = new LocalLoginContext(applicationName,cbh,configuration);
                        }else{
                            loginContext = new LoginContext(applicationName,cbh);
                        }

                } catch(LoginException se) {
                      logger.log(Level.SEVERE,"LoginContext cannot be created. "+se.getMessage(),se);
                      throw se;
                }

                try{
                        if(local){
                                localLoginContext.login();
                        }else{
                                loginContext.login();
                        }
                        
                        //authentication succeed
                        authenticationSucceed = true;
                     
                }catch(LoginException le){
                    authenticationSucceed = false;
                    Throwable localizedThrowable = ThrowableUtils.localizeThrowable(le,locale);
                    throw (LoginException)localizedThrowable;
                }finally{
                   AuthenticationManager authNManager = AuthenticationManagerFactory.getAuthenticationManager();
                   try {
                	   long now = System.currentTimeMillis();
                       authNManager.persistUserAccessAttempt(nameCallback.getName(),inetCbk.getHostAdress(),inetCbk.getHostName(),authenticationSucceed,now);
                       logger.info("user from Host adress="+inetCbk.getHostAdress()+" bound to host name="+inetCbk.getHostName()+" has tried to authenticate. boolean result ="+authenticationSucceed+" timeStamp="+now+" locale="+locale.getDisplayName());
                   } catch (AuthenticationException ex) {
                       logger.log(Level.SEVERE,ex.getMessage(),ex);
                       throw new LoginException(ex.getMessage());
                   }
                }

                if(local){
                        subject = localLoginContext.getSubject();
                }else{
                        subject = loginContext.getSubject();
                }
                if(subject != null){
                        // used in ABAC permissions
                        UserPrincipal userPrincipal = new UserPrincipal(subject);
                        subject.getPrincipals().add(userPrincipal);
                }
	}


	
	
	 /**
     * retrieve the subject from the loginContext.
     * @return authenticated Subject, otherwise <strong>null</strong>.
     */
    public Subject getSubject(){
    	return subject;
    }
    
    /**
     * logout the user with the related LoginContext.
     */
    public  void logout() {
      if(loggedOut==false){
    	 try {
    		if(local){
	    			if(localLoginContext!=null){
	    				localLoginContext.logout();
		                loggedOut = true;
		         	}else{
		         		logger.finest(" user is not logged, so we don't logout him ");
		         	}
    		}else{
		         	if(loginContext!=null){
		                loginContext.logout();
		                loggedOut = true;
		         	}else{
		         		logger.finest(" user is not logged, so we don't logout him ");
		         	}
    		}
         } catch (LoginException e) {
             logger.log(Level.SEVERE," error raised when the user logout "+e.getMessage(),e);
         }
      }
    }

	public boolean isLocal() {
		return local;
	}
	
	
}
