/*
jGuard is a security framework based on top of jaas (java authentication and authorization security).
it is written for web applications, to resolve simply, access control problems.
version $Name$
http://sourceforge.net/projects/jguard/

Copyright (C) 2004  Charles GAY

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


jGuard project home page:
http://sourceforge.net/projects/jguard/

*/

package net.sf.jguard.core.authentication;

import java.util.logging.Level;
import java.util.logging.Logger;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.login.LoginException;
import net.sf.jguard.core.CoreConstants;

/**
 * Authenticate a user in an application with a specific AuthenticationBindings.
 * @author <a href="mailto:diabolo512@users.sourceforge.net">Charles Gay</a>
 * @since 1.1
 */
public class AuthenticationServicePoint {

    
    private static final Logger logger = Logger.getLogger(AuthenticationServicePoint.class.getName());
    
    /**
      * authenticate the user.
      * if the {@link AuthenticationBindings} implementation implements the
      * Stateful interface, this method removes a possible old {@link AuthenticationUtils} object,
      * invalidate the session, create a new one, and bound the new Authenticationutils to it.
      */
     public static boolean authenticate(AccessContext context, String applicationName , AuthenticationBindings authNBindings){
        authNBindings.setRequestAttribute(context,CoreConstants.REGISTRATION_DONE, Boolean.valueOf(false));
        CallbackHandler cbh = authNBindings.getCallbackHandler(context);
        
        try {
            AuthenticationUtils authNUtils = authNBindings.getAuthenticationUtils(context);
            authNUtils.login(applicationName, cbh);
            //for security reasons(man-in-the-middle attack for sessionID cookie),
            //we remove the old session if authentication is successful
            //and create a new session
            if(authNBindings instanceof Stateful){
                 //we remove this variable before invalidate the session
                //to prevent the session listener to erase the subject
                ((Stateful)authNBindings).removeSessionAttribute(context,CoreConstants.AUTHN_UTILS);
                ((Stateful)authNBindings).invalidateSession(context);
            }
            boolean result = authNBindings.authenticationSucceed(context);
             if(authNBindings instanceof Stateful){
                 ((Stateful)authNBindings).setSessionAttribute(context, CoreConstants.AUTHN_UTILS, authNUtils);
             }
            return result;
        } catch (LoginException e) {
            logger.log(Level.FINEST,"authentication failed "+e.getMessage(),e);
            String messageError = null;
            messageError = e.getLocalizedMessage();
            //we store in the user' session the reason the authentication failed
            authNBindings.setRequestAttribute(context,CoreConstants.LOGIN_EXCEPTION_MESSAGE,messageError);
            authNBindings.setRequestAttribute(context,CoreConstants.LOGIN_EXCEPTION_CLASS,e.getClass());
            return authNBindings.authenticationFailed(context);
        }
    }
     
     /**
      * impersonate the current user with the provided callbacks.
      * @param context
      * @param applicationName
      * @param authenticationBindings
      * @param callbacks
      * @return
      */
     public static boolean authenticateWithImpersonation(AccessContext context,String applicationName,AuthenticationBindings authenticationBindings,Callback[] callbacks) {
         AuthenticationBindings impersonatedAuthenticationBindings = null;
	if(authenticationBindings instanceof Stateful){
            impersonatedAuthenticationBindings = new StatefulImpersonationAuthenticationBindings(authenticationBindings,callbacks);
        }else{	
            impersonatedAuthenticationBindings = new ImpersonationAuthenticationBindings(authenticationBindings,callbacks);
        }
		return AuthenticationServicePoint.authenticate(context,applicationName,impersonatedAuthenticationBindings);
     }

     /**
      * impersonate the current user as a Guest user with the related credentials.
      * @param context
      * @param applicationName
      * @param authenticationBindings
      * @return
      */
     public static boolean authenticateAsGuest(AccessContext context,String applicationName,AuthenticationBindings authenticationBindings) {
		
		Callback[] callbacks = new Callback[2];
		NameCallback nameCallback = new NameCallback("login");
		nameCallback.setName(CoreConstants.GUEST);
		callbacks[0] = nameCallback;
		
		PasswordCallback pwdCbk =new PasswordCallback("password",false);
		pwdCbk.setPassword(CoreConstants.GUEST.toCharArray());
		callbacks[1] = pwdCbk;
		
		return authenticateWithImpersonation(context,applicationName,authenticationBindings,callbacks);
     }
	
     
     

}
