package net.sf.javaprinciples.data.visitor;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

/**
 * Invoke the operational method on the target
 *
 * @author Warwick Slade
 */
public class ReflectiveOperation<T, R> implements Operation
{
    private T target;
    private ResultPolicy<R> resultPolicy;

    /**
     * Invoke the operational method on the target.
     *
     * @param visitor
     */
    public void visit(Visitor visitor)
    {
        // Determine the operational method
        Method method = determineMethod();

        // Determine the parameters
        Object[] params = determineParams(method, visitor);

        // Invoke
        Object result = invokeMethod(method, params);

        // Store result
        storeResult(method, result, visitor);
    }

    private void storeResult(Method method, Object result, Visitor visitor)
    {
        if (method.getReturnType() != void.class)
        {
            resultPolicy.result(visitor, (R)result);
        }
    }

    private Object invokeMethod(Method method, Object[] params)
    {
        try
        {
            return method.invoke(target, params);
        }
        catch (IllegalAccessException e)
        {
            throw new RuntimeException(
                    String.format("Illegal access to method %s on the class %s", method.getName(), target,getClass().getName()), e);
        }
        catch (InvocationTargetException e)
        {
            throw new RuntimeException(
                    String.format("Invocation issue on method %s on the class %s", method.getName(), target,getClass().getName()), e);
        }
    }

    private Object[] determineParams(Method method, Visitor visitor)
    {
        Class[] paramTypes = method.getParameterTypes();
        if (paramTypes.length == 0)
        {
            return new Object[0];
        }

        Object[] operands = visitor.getOperands();
        if (paramTypes.length > operands.length)
        {
            throw new RuntimeException(String.format("The number of parameters %s is greater than the number of operands %s",
                    paramTypes.length,
                    operands.length));
        }

        Object[] params = new Object[paramTypes.length];

        System.arraycopy(operands, 0, params, 0, params.length);
        return params;
    }

    private Method determineMethod()
    {
        Method[] methods = target.getClass().getMethods();
        for (Method method : methods)
        {
            if (Modifier.isPublic(method.getModifiers()))
            {
                String name = method.getName();
                if (!name.startsWith("get") && !name.startsWith("set") && !name.startsWith("is"))
                {
                    return method;
                }
            }
        }
        throw new RuntimeException(String.format("The class %s does not have an operation method", target,getClass().getName()));
    }

    public void setTarget(T target)
    {
        this.target = target;
    }

    public void setResultPolicy(ResultPolicy<R> resultPolicy)
    {
        this.resultPolicy = resultPolicy;
    }
}
