package net.sf.javaprinciples.data.transformer.spring;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.springframework.core.convert.converter.Converter;
import org.springframework.core.convert.converter.ConverterFactory;
import org.springframework.util.ClassUtils;

/**
 * A copy of the spring class StringToEnumConverterFactory with the added bonus of handling JAXB enums
 *
 * @author Warwick Slade
 */
@SuppressWarnings("unchecked")
public final class StringToJaxbEnumConverterFactory implements ConverterFactory<String, Enum>
{
    public <T extends Enum> Converter<String, T> getConverter(Class<T> targetType)
    {
        return new StringToEnum(targetType);
    }

    private class StringToEnum<T extends Enum> implements Converter<String, T>
    {
        private final Class<T> enumType;
        private final Method fromValueMethod;

        public StringToEnum(Class<T> enumType)
        {
            this.enumType = enumType;
            fromValueMethod = ClassUtils.getMethodIfAvailable(enumType, "fromValue", new Class[]{String.class});
        }

        public T convert(String source)
        {
            if (source.length() == 0)
            {
                // It's an empty enum identifier: reset the enum value to null.
                return null;
            }
            if (fromValueMethod == null)
            {
                return (T) Enum.valueOf(this.enumType, source.trim());
            }
            else
            {
                try
                {
                    return (T)fromValueMethod.invoke(null, source);
                }
                catch (IllegalAccessException e)
                {
                    return null;
                }
                catch (InvocationTargetException e)
                {
                    return null;
                }
            }
        }
    }

}
