package net.sf.javaprinciples.data.transformer;

import java.util.List;

import org.omg.uml.Enumeration;
import org.omg.uml.Property;
import org.springframework.beans.InvalidPropertyException;
import org.springframework.beans.PropertyAccessException;

import net.sf.jcc.model.parser.UnexpectedException;
import net.sf.jcc.model.parser.uml2.ModelElement;

/**
 * Default implementation of a ModelElementSourceDestinationMapper.
 * Handles the retrieval of source attribute names and values and
 * the setting of these values on the destination attribute.
 *
 * @author Kay Chevalier
 */
public class SourceDestinationAttributeMapper implements ModelElementSourceDestinationMapper
{
    protected ObjectTypeMapper objectTypeMapper;
    private ModelElement mappingElement;

    /**
     * Given a list of {@link ModelElement} objects, retrieves the element name
     * from the first model element in the list.
     *
     * @param modelElements - a list of ModelElements to retrieve the attribute name from.
     * @return - the attribute name
     */
    @Override
    public String getAttributeName(List<ModelElement> modelElements)
    {
        if (modelElements.isEmpty())
            return null;

        ModelElement modelElement = modelElements.get(0);

        return getAttributeName(modelElement);
    }

    /**
     * Given a {@link ModelElement} object, retrieves the element name
     * from this model element.
     *
     * @param modelElement - a ModelElement to retrieve the attribute name from.
     * @return - the attribute name
     */
    public String getAttributeName(ModelElement modelElement)
    {
        if(modelElement != null && modelElement.getElement() instanceof Property)
        {
            return ((Property)modelElement.getElement()).getName();
        }
        else if(modelElement != null && modelElement.getElement() instanceof org.omg.uml.Class)
        {
            return ((org.omg.uml.Class)modelElement.getElement()).getName();
        }
        else if (modelElement != null && modelElement.getElement() instanceof Enumeration)
        {
            return ((Enumeration)modelElement.getElement()).getName();
        }

        return "";
    }

    /**
     * Given a list of {@link ModelElement} objects, retrieves the attribute name from the
     * first model element in the list. This attribute name is then used to find the
     * attribute of the parent object in which the sourceAttributeValue should be set.
     *
     * @param modelElements - a list of ModelElements to retrieve the parent object attribute name from.
     * @param sourceAttributeValue - the value to be set on the parent object
     * @param parentObject - the object to set the attribute value on.
     */
    @Override
    public void setAttributeOnParentObject(List<ModelElement> modelElements, Object sourceAttributeValue, Object parentObject)
    {
        for (ModelElement modelElement : modelElements)
        {
            String destinationAttributeName = getAttributeName(modelElement);

            if (destinationAttributeName != null)
            {
                char[] stringArray = destinationAttributeName.toCharArray();
                stringArray[0] = Character.toLowerCase(stringArray[0]);
                destinationAttributeName = new String(stringArray);
            }

            try
            {
                objectTypeMapper.assignAttributeToObject(sourceAttributeValue, parentObject, destinationAttributeName);
            }
            catch (InvalidPropertyException e)
            {
                throw new UnexpectedException(String.format("Could not set property %s on %s . Message: %s",
                        destinationAttributeName, parentObject.getClass().getName(), e.getMessage()),e.getCause());
            }
            catch (PropertyAccessException e)
            {
                throw new UnexpectedException(String.format("Could not set property %s on %s . Message: %s",
                        destinationAttributeName, parentObject.getClass().getName(), e.getMessage()),e.getCause());
            }
        }
    }

    /**
     * Given a list of {@link ModelElement} objects, retrieves the attribute name from the
     * first model element in the list. This attribute name is then used to find the corresponding
     * attribute value from the parent object.
     * @param modelElements - a list of ModelElements to retrieve the parent object attribute name from.
     * @param parentObject - the object to retrieve the attribute value from.
     * @return - the attribute object
     */
    @Override
    public Object getAttributeFromObject(List<ModelElement> modelElements, Object parentObject)
    {
        String attributeName = getAttributeName(modelElements);
        return objectTypeMapper.getSourceAttribute(parentObject, attributeName);
    }

    public String getMappingElementAttributeName()
    {
        return ModelElementMapperHelper.getPropertyNameFromElement(mappingElement);
    }

    public void setObjectTypeMapper(ObjectTypeMapper objectTypeMapper)
    {
        this.objectTypeMapper = objectTypeMapper;
    }


    public ModelElement getMappingElement()
    {
        return mappingElement;
    }

    public void setMappingElement(ModelElement mappingElement)
    {
        this.mappingElement = mappingElement;
    }
}
