package net.sf.javaprinciples.data.transformer;

import java.util.List;

import org.omg.uml.*;
import org.omg.uml.Class;

import au.com.sparxsystems.Attribute;
import au.com.sparxsystems.Element;
import au.com.sparxsystems.Tag;
import net.sf.javaprinciples.core.ObjectClassHelper;
import net.sf.javaprinciples.core.UnexpectedException;
import net.sf.jcc.model.parser.uml2.ElementStore;
import net.sf.jcc.model.parser.uml2.ModelElement;

/**
 *  Provides utility methods relevant to the {@link ModelElement} class.
 *
 * @author Kay Chevalier
 */
public class ModelElementMapperHelper
{
    private static final String TARGET_NAMESPACE_TAG_NAME = "targetNamespace";

    /**
     * Retrieves a formatted package name from the model elements parent.  Escalates to the parents parent
     * until a package name is found or no parent exists.
     *
     * @param modelElement to retrieve the package name from
     * @param store to retrieve any parent model elements from
     * @return  packageName String
     */
    public static String getPackageNameFromParentElement(ModelElement modelElement, ElementStore store)
    {
        String parentGuid = modelElement.getParentGuid();
        if (parentGuid == null)
            throw new UnexpectedException(String.format("Unable to find package name for %s. Parent Guid not found.",
                    modelElement.getGuid()));

        String packageName;
        ModelElement parentModelElement = store.get(parentGuid);
        packageName = ModelElementMapperHelper.retrieveModelElementTagValue(parentModelElement, TARGET_NAMESPACE_TAG_NAME);

        if (packageName == null || "".equals(packageName))
            return getPackageNameFromParentElement(parentModelElement, store);

        return packageName;
    }

    public static String retrieveModelElementTagValue(ModelElement modelElement, String tagName)
    {
        Object extension = modelElement.getExtension();
        List<Tag> tags = null;

        if (extension instanceof Element)
            tags = ((Element)extension).getTags();
        else if (extension instanceof Attribute)
            tags = ((Attribute)extension).getTags();

        if (tags != null)
        {
            for (Tag tag : tags)
            {
                if (tagName.equals(tag.getName()))
                {
                    return tag.getValue();
                }
            }
        }
        return null;
    }

    /**
     * Creates an object instance derived from the model element name and package.
     *
     * @param modelElement to retrieve the name and package from
     * @param store - the store to retrieve any parent model elements from
     * @return an instance of the ModelElement represented Class
     */
    public static Object createObjectFromClassModelElement(ModelElement modelElement, ElementStore store,
                                                           ObjectTypeMapper objectTypeMapper)
    {
        String packageName = getPackageNameFromParentElement(modelElement, store);
        packageName = objectTypeMapper.formatPackageName(packageName);
        StringBuilder packageNameBuilder = new StringBuilder();
        packageNameBuilder.append(packageName.toLowerCase());
        packageNameBuilder.append(".");
        packageNameBuilder.append(((org.omg.uml.Class)modelElement.getElement()).getName());

        return ObjectClassHelper.createObjectFromClassName(packageNameBuilder.toString());
    }

    /**
     * Retrieves the name from the {@link ModelElement}
     * @param modelElement - to retrieve the name from
     * @return ModelElement name
     */
    public static String getPropertyNameFromElement(ModelElement modelElement)
    {
        if (modelElement == null)
            return null;

        if (modelElement.getElement() instanceof Property)
        {
            return ((Property)modelElement.getElement()).getName();
        }
        else if (modelElement.getElement() instanceof org.omg.uml.Class)
        {
            return ((Class)modelElement.getElement()).getName();
        }

        throw new net.sf.jcc.model.parser.UnexpectedException(String.format("Unable to get property name from model element %s. " +
                "Model element class type %s not supported",  modelElement.getGuid(), modelElement.getElement().getClass().getName()));

    }

    public static String getAttributeTypeFromProperties(ModelElement modelElement)
    {
        Object extension = modelElement.getExtension();
        if (extension instanceof Attribute)
        {
            return ((Attribute)extension).getProperties().getType();
        }

        return null;
    }
}
