package net.sf.javaprinciples.data.transformer;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.omg.uml.Property;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import au.com.sparxsystems.Association;
import au.com.sparxsystems.Attribute;
import au.com.sparxsystems.Element;
import net.sf.jcc.model.parser.UnexpectedException;
import net.sf.jcc.model.parser.uml2.ElementStore;
import net.sf.jcc.model.parser.uml2.ModelElement;

/**
 * Factory to create a {@link ModelElementMapper} from the stereotype of a given
 * model element.
 *
 * @author Kay Chevalier
 */
public class ModelElementMapperFactory<T,U> extends ModelElementFactory implements MapperFactory
{
    private static final String RIGHT_ATTRIBUTE_MAPPER_TAG = "rightAttributeMapper";
    private static final String LEFT_ATTRIBUTE_MAPPER_TAG = "leftAttributeMapper";
    public static final String IMPLEMENTATION_TAG = "implementation";
    private static final String DEFAULT_SOURCE_ATTRIBUTE_MAPPER = "defaultSourceAttributeMapper";

    private MapperBeanFactory<T,U> mapperBeanFactory;
    private SourceDestinationAttributeMapperBeanFactory sourceDestinationAttributeMapperBeanFactory;
    private Logger log = LoggerFactory.getLogger(ModelElementMapperFactory.class);

    @Override
    public ModelElementMapper<T,U> createMapper(String mapperGuid, ElementStore store, TransformerFactory.DIRECTION_TYPE direction)
    {
        ModelElement modelElement = store.get(mapperGuid);
        String stereotype = getModelElementStereotype(modelElement);
        String implementationTag = ModelElementMapperHelper.retrieveModelElementTagValue(modelElement, IMPLEMENTATION_TAG);

        ModelElementMapper<T,U> modelElementMapper = getModelElementMapper(StringUtils.isEmpty(implementationTag) ? stereotype : implementationTag);
        modelElementMapper.setObjectTypeMapper(objectTypeMapper);
        ModelElement sourceModelElement = retrieveSourceFromAssociations(modelElement, direction, store);
        ModelElement destinationModelElement = retrieveDestinationFromAssociations(modelElement, direction, store);
        modelElementMapper.setSourceModelElement(sourceModelElement);
        modelElementMapper.setDestinationModelElement(destinationModelElement);

        if (modelElementMapper instanceof ComplexMapper || modelElementMapper instanceof PostProcessorMapper)
        {
            if (sourceModelElement == null && destinationModelElement == null)
            {
                throw new UnexpectedException(String.format("You must have at least one source or target model element for %s"
                        , modelElement.getGuid()));
            }

            modelElementMapper.setModelElementMappers(retrieveMappersFromAssociations(modelElement, store, direction));
            modelElementMapper.setPostProcessorMappers(retrievePostProcessorMappersFromAssociations(modelElement, store, direction));

            log.info("Creating Mapper: {}, {}, Source Model Element: {}, Target Model Element: {} ",
                   new Object[] {modelElementMapper.getClass().getName(), mapperGuid ,sourceModelElement == null ? null : sourceModelElement.getGuid(),
                           destinationModelElement == null ? null : destinationModelElement.getGuid()});
        }
        else if (modelElementMapper instanceof AttributeMapper)
        {
            List<ModelElementMapper> attributeMapperList = createAttributeMapperList(store, modelElement, direction, objectTypeMapper);
            modelElementMapper.setModelElementMappers(attributeMapperList);
        }

        return modelElementMapper;
    }

    private List<ModelElementMapper> createAttributeMapperList(ElementStore store, ModelElement mapperModelElement,
                                                               TransformerFactory.DIRECTION_TYPE direction, ObjectTypeMapper objectTypeMapper)
    {
        ArrayList<ModelElementMapper> attributeMapperList = new ArrayList<ModelElementMapper>();

        List<Property> mapperModelAttributes = ((org.omg.uml.Class)mapperModelElement.getElement()).getOwnedAttribute();
        for (Property mapperProperty : mapperModelAttributes)
        {
            ModelElement mapperPropertyElement = store.reverse(mapperProperty);
            List<Association> mapperPropertyAssociations = mapperPropertyElement.getAssociations();
            if (mapperPropertyAssociations == null)
            {
                continue;
            }

            List<ModelElement> sourceModelElements = new ArrayList<ModelElement>();
            List<ModelElement> destinationModelElements = new ArrayList<ModelElement>();

            populateSourceAndDestinationModelElements(store, direction, mapperPropertyAssociations, sourceModelElements, destinationModelElements);

            if (destinationModelElements.isEmpty() && sourceModelElements.isEmpty())
            {
                throw new UnexpectedException("You must have at least one source or target model element for %s"
                        + mapperPropertyElement.getGuid());
            }

            String modelElementStereotype = getModelElementStereotype(mapperPropertyElement);
            ModelElementMapper<T,U> attributeMapper = null;
            if (modelElementStereotype != null)
                attributeMapper = getModelElementMapper(modelElementStereotype);

            if (attributeMapper == null)
            {
                attributeMapper = new DefaultAttributeMapper();
            }

            attributeMapper.setSourceModelElements(sourceModelElements);
            attributeMapper.setDestinationModelElements(destinationModelElements);
            attributeMapper.setStore(store);
            attributeMapper.setObjectTypeMapper(objectTypeMapper);
            attributeMapperList.add(attributeMapper);


            if (attributeMapper instanceof DefaultAttributeMapper)
            {
                if(destinationModelElements.isEmpty())
                {
                    throw new UnexpectedException("You must have at least one target model element for %s"
                            + mapperPropertyElement.getGuid());
                }

                SourceDestinationAttributeMapper sourceAttributeMapper = getModelElementSourceDestinationAttributeMapper(
                        mapperPropertyElement, direction, true);
                sourceAttributeMapper.setObjectTypeMapper(objectTypeMapper);
                SourceDestinationAttributeMapper destinationAttributeMapper = getModelElementSourceDestinationAttributeMapper(
                        mapperPropertyElement, direction, false);
                destinationAttributeMapper.setObjectTypeMapper(objectTypeMapper);
                ((DefaultAttributeMapper)attributeMapper).setSourceModelElementMapper(sourceAttributeMapper);
                ((DefaultAttributeMapper)attributeMapper).setDestinationModelElementMapper(destinationAttributeMapper);
            }


            String sourceModelElementGuid = sourceModelElements.isEmpty() ? null : sourceModelElements.get(0).getGuid();
            String destinationModelElementGuid = destinationModelElements.isEmpty() ? null : destinationModelElements.get(0).getGuid();
            log.info("Creating Mapper: {}, {}, Source Model Element: {}, Target Model Element: {} ",
                    new Object[] {attributeMapper.getClass().getName(), mapperPropertyElement.getGuid() ,
                            sourceModelElementGuid, destinationModelElementGuid});
        }

        if (attributeMapperList.isEmpty())
        {
            throw new UnexpectedException(String.format("Unable to perform transformation for transformer %s . " +
                    "No associated mappers found.", mapperModelElement.getGuid()));
        }

        return attributeMapperList;
    }

    private void populateSourceAndDestinationModelElements(ElementStore store, TransformerFactory.DIRECTION_TYPE direction, List<Association> mapperPropertyAssociations, List<ModelElement> sourceModelElements, List<ModelElement> destinationModelElements)
    {
        for (Association association : mapperPropertyAssociations)
        {
            String associationStereotype = association.getProperties().getStereotype();
            String targetId = association.getTargetAttributeId() != null ? association.getTargetAttributeId() : association.getTargetClassId();
            ModelElement attributeModelElement = store.get(targetId);

            if (TransformerFactory.DIRECTION_TYPE.leftToRight.equals(direction))
            {
                if (RIGHT_STEREOTYPE_PROPERTY.equals(associationStereotype))
                {
                    destinationModelElements.add(attributeModelElement);
                }
                else if (LEFT_STEREOTYPE_PROPERTY.equals(associationStereotype))
                {
                   sourceModelElements.add(attributeModelElement);
                }
            }
            else
            {
                if (LEFT_STEREOTYPE_PROPERTY.equals(associationStereotype))
                {
                    destinationModelElements.add(attributeModelElement);
                }
                else if (RIGHT_STEREOTYPE_PROPERTY.equals(associationStereotype))
                {
                    sourceModelElements.add(attributeModelElement);
                }
            }
        }
    }

    private SourceDestinationAttributeMapper getModelElementSourceDestinationAttributeMapper(ModelElement modelElement, TransformerFactory.DIRECTION_TYPE direction, boolean isSource)
    {
        String sourceDestinationAttributeMapper;

        if(isSource)
        {
            if (TransformerFactory.DIRECTION_TYPE.leftToRight.equals(direction))
                sourceDestinationAttributeMapper = ModelElementMapperHelper.retrieveModelElementTagValue(modelElement, LEFT_ATTRIBUTE_MAPPER_TAG);
            else
                sourceDestinationAttributeMapper = ModelElementMapperHelper.retrieveModelElementTagValue(modelElement, RIGHT_ATTRIBUTE_MAPPER_TAG);
        }
        else
        {
            if (TransformerFactory.DIRECTION_TYPE.leftToRight.equals(direction))
                sourceDestinationAttributeMapper = ModelElementMapperHelper.retrieveModelElementTagValue(modelElement, RIGHT_ATTRIBUTE_MAPPER_TAG);
        else
                sourceDestinationAttributeMapper = ModelElementMapperHelper.retrieveModelElementTagValue(modelElement, LEFT_ATTRIBUTE_MAPPER_TAG);
        }

        SourceDestinationAttributeMapper sourceDestinationMapper;
        if (sourceDestinationAttributeMapper == null)
        {
            sourceDestinationAttributeMapper = DEFAULT_SOURCE_ATTRIBUTE_MAPPER;
        }
        sourceDestinationMapper = getSourceDestinationMapper(sourceDestinationAttributeMapper);

        sourceDestinationMapper.setObjectTypeMapper(objectTypeMapper);
        sourceDestinationMapper.setMappingElement(modelElement);

        return sourceDestinationMapper;
    }

    private SourceDestinationAttributeMapper getSourceDestinationMapper(String mapperName)
    {
        return sourceDestinationAttributeMapperBeanFactory.createModelElementMapper(mapperName);
    }

    private Mapper<T,U> getRegisteredMapper(String mapperName)
    {
        return mapperBeanFactory.createMapper(mapperName);
    }


    private ModelElementMapper<T,U> getModelElementMapper(String mapperName)
    {
        Object mapper = getRegisteredMapper(mapperName);

        if (mapper instanceof ModelElementMapper)
            return (ModelElementMapper<T,U>)mapper;
        else
            throw new UnexpectedException(String.format("Expected a ModelElementMapper type for mapperName: %s", mapperName));
    }

    private String getModelElementStereotype(ModelElement destinationMapper)
    {
        if (destinationMapper == null || destinationMapper.getExtension() == null)
            return null;

       Object extension = destinationMapper.getExtension();
        if (extension instanceof Element)
            return ((Element)extension).getProperties().getStereotype();
        else if (extension instanceof Attribute)
            return ((Attribute)extension).getStereotype().getStereotype();

        return null;
    }

    public void setMapperBeanFactory(MapperBeanFactory<T, U> mapperBeanFactory)
    {
        this.mapperBeanFactory = mapperBeanFactory;
    }

    public void setSourceDestinationAttributeMapperBeanFactory(SourceDestinationAttributeMapperBeanFactory sourceDestinationAttributeMapperBeanFactory)
    {
        this.sourceDestinationAttributeMapperBeanFactory = sourceDestinationAttributeMapperBeanFactory;
    }
}
