package net.sf.javaprinciples.data.transformer;

import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;

import org.omg.uml.*;
import org.omg.uml.Class;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import au.com.sparxsystems.Attribute;
import net.sf.jcc.model.parser.UnexpectedException;
import net.sf.jcc.model.parser.uml2.ElementStore;
import net.sf.jcc.model.parser.uml2.ModelElement;

/**
 * Abstract class for model element mappers.
 * @author Kay Chevalier
 */
public abstract class ModelElementMapper <T, U> implements Mapper<T, U>
{
    protected ElementStore store;
    protected List<ModelElement> sourceModelElements;
    protected List<ModelElement> destinationModelElements;
    private List<ModelElementMapper> modelElementMappers;
    protected ObjectTypeMapper objectTypeMapper;
    protected List<PostProcessorMapper> postProcessorMappers;
    private static final Logger log = LoggerFactory.getLogger(ModelElementMapper.class);

    public List<ModelElementMapper> getModelElementMappers()
    {
        return modelElementMappers;
    }

    public void setModelElementMappers(List<ModelElementMapper> modelElementMappers)
    {
        this.modelElementMappers = modelElementMappers;
    }

    public ModelElement getDestinationModelElement()
    {
        return getPrimaryDestinationModelElement();
    }

    public void setDestinationModelElement(ModelElement destinationModelElement)
    {
        addDestinationModelElement(destinationModelElement);
    }

    public ModelElement getPrimaryDestinationModelElement()
    {
        if (CollectionUtils.isEmpty(destinationModelElements))
        {
            return null;
        }
        return destinationModelElements.get(0);
    }

    public ModelElement getPrimarySourceModelElement()
    {
        if (CollectionUtils.isEmpty(sourceModelElements))
        {
            return null;
        }
        return sourceModelElements.get(0);
    }

    public ModelElement getSourceModelElement()
    {
        return getPrimarySourceModelElement();
    }

    public void setSourceModelElement(ModelElement sourceModelElement)
    {
        addSourceModelElement(sourceModelElement);
    }

    public void addSourceModelElement(ModelElement modelElement)
    {
        if (sourceModelElements == null)
            sourceModelElements = new ArrayList<ModelElement>();

        sourceModelElements.add(modelElement);
    }

    public void addDestinationModelElement(ModelElement modelElement)
    {
        if (destinationModelElements == null)
            destinationModelElements = new ArrayList<ModelElement>();

        destinationModelElements.add(modelElement);
    }

    public void setStore(ElementStore store)
    {
        this.store = store;

    }

    protected String getAttributeTypeFromModelElement(ModelElement destinationModelElement)
    {
        if (destinationModelElement.getElement() instanceof Property)
        {
            String ref = destinationModelElement.getReference(new QName("type"));
            ModelElement propertyTypeElement = store.get(ref);
            if (propertyTypeElement == null)
                return null;

            String packageName =  ModelElementMapperHelper.getPackageNameFromParentElement(propertyTypeElement, store);
            packageName = objectTypeMapper.formatPackageName(packageName);

            String attributeType = ((Attribute)destinationModelElement.getExtension()).getProperties().getType();
            return packageName + "." + attributeType;
        }
        else if (destinationModelElement.getElement() instanceof org.omg.uml.Class)
        {
            String classPackageName = ModelElementMapperHelper.getPackageNameFromParentElement(destinationModelElement, store);
            classPackageName = objectTypeMapper.formatPackageName(classPackageName);
            return classPackageName + "." + ((Class)destinationModelElement.getElement()).getName();
        }

        throw new UnexpectedException(String.format("Unable to determine attribute type from model element %s. " +
                "Model element class type %s not supported",  destinationModelElement.getGuid(), destinationModelElement.getElement().getClass().getName()));
    }

    protected boolean processMapperAssociations(T input, U output)
    {
        if (CollectionUtils.isEmpty(getModelElementMappers()))
        {
            return false;
        }

        boolean attributeSet = false;
        for (Mapper mapper : getModelElementMappers())
        {
            try
            {
                mapper.map(input, output);
                attributeSet = true;
            } catch (AttributeNotFoundException e)
            {
                continue;
            }
        }

        return attributeSet;
    }

    protected T retrieveInput(ModelElement sourceModelElement, T input)
    {
        if (sourceModelElement == null)
            return input;

        String inputClassName = null;
        if (sourceModelElement.getElement() instanceof Class)
        {
            Class inputClass = (Class)sourceModelElement.getElement();
            inputClassName = inputClass.getName();
        }
        else if (sourceModelElement.getElement() instanceof Property)
        {
            Property inputClass = (Property)sourceModelElement.getElement();
            inputClassName = inputClass.getName();
        }

        return (T)objectTypeMapper.getSourceAttribute(input, inputClassName);
    }

    public void setObjectTypeMapper(ObjectTypeMapper objectTypeMapper)
    {
        this.objectTypeMapper = objectTypeMapper;
    }

    protected void logAttributeMapping(String sourceClass, String sourceAttribute, String targetClass, String targetAttribute, Object value)
    {
        log.info("{}, {}, {}, {}, {}, {}",
                new Object[] {this.getClass().getName(), sourceClass, sourceAttribute, targetClass, targetAttribute, value});
    }

    public List<ModelElement> getDestinationModelElements()
    {
        return destinationModelElements;
    }

    public void setDestinationModelElements(List<ModelElement> destinationModelElements)
    {
        this.destinationModelElements = destinationModelElements;
    }

    public List<ModelElement> getSourceModelElements()
    {
        return sourceModelElements;
    }

    public void setSourceModelElements(List<ModelElement> sourceModelElements)
    {
        this.sourceModelElements = sourceModelElements;
    }

    protected U instantiateOutputFromAttributeName(U output, String attributeName)
    {
        return (U)objectTypeMapper.instantiateObjectFromAttributeName(output, attributeName);
    }

    /**
     * Performs post processing mapping once all the mappers have been called.
     * Calls a map on the PostProcessorMapper instances to perform any additional processing
     * required.  Does not instantiate any object instances as all have been assumed to be
     * created by the first mapping pass through.
     *
     * @param input
     * @param output
     */
    protected void performPostProcessingMapping(T input, U output)
    {
        List<PostProcessorMapper> postProcessorMappers = getPostProcessorMappers();
        if (CollectionUtils.isEmpty(postProcessorMappers))
        {
            return;
        }

        U destinationOutput = null;

        if(getDestinationModelElement() != null)
        {
            String attributeName =  ModelElementMapperHelper.getPropertyNameFromElement(getDestinationModelElement());
            destinationOutput = (U)objectTypeMapper.getAttributeFromObject(output, attributeName);
        }


        for (Mapper postProcessingMapper : postProcessorMappers)
        {
            postProcessingMapper.map(input, destinationOutput == null ? output : destinationOutput);
        }
    }

    public List<PostProcessorMapper> getPostProcessorMappers()
    {
        return postProcessorMappers;
    }

    public void setPostProcessorMappers(List<PostProcessorMapper> postProcessorMappers)
    {
        this.postProcessorMappers = postProcessorMappers;
    }
}
