package net.sf.javaprinciples.data.transformer;

import java.util.ArrayList;
import java.util.List;

import au.com.sparxsystems.Association;
import net.sf.jcc.model.parser.UnexpectedException;
import net.sf.jcc.model.parser.uml2.ElementStore;
import net.sf.jcc.model.parser.uml2.ModelElement;

/**
 * Abstract class to implement common functionality in creating transformers or mappers from model elements.
 * @author Kay Chevalier
 */
public abstract class ModelElementFactory
{
    protected static final String RIGHT_STEREOTYPE_PROPERTY = "right";
    protected static final String LEFT_STEREOTYPE_PROPERTY = "left";
    public static final String SOURCE_DESTINATION_DIRECTION_PROPERTY = "Source -> Destination";
    public static final String POST_PROCESSING_STEREOTYPE_PROPERTY = "postProcessor";
    protected ObjectTypeMapper objectTypeMapper;

    protected ModelElement retrieveSourceFromAssociations(ModelElement transformerModelElement,
                                                          TransformerFactory.DIRECTION_TYPE direction, ElementStore store)
    {
        return retrieveModelElementFromAssociations(true, transformerModelElement, direction, store);
    }

    protected ModelElement retrieveDestinationFromAssociations(ModelElement transformerModelElement,
                                                               TransformerFactory.DIRECTION_TYPE direction, ElementStore store)
    {
        return retrieveModelElementFromAssociations(false, transformerModelElement, direction, store);
    }

    /**
     * Retrieves the {@link ModelElement} from the associations of a given transformerModelElement.
     * Returns a left or right stereotyped association depending on the {@link net.sf.javaprinciples.data.transformer.TransformerFactory.DIRECTION_TYPE}
     * and the isSourceAssociation provided.
     *
     * @param isSourceAssociations - indicator specifying if it is the source or destination association we are retrieving.
     * @param transformerModelElement - the model element to retrieve the associations from
     * @param direction - the direction of the transformation
     * @param store - the store to retrieve the associations from
     * @return  the left or right associated ModelElement
     */
    protected ModelElement retrieveModelElementFromAssociations(boolean isSourceAssociations, ModelElement transformerModelElement,
                                                                TransformerFactory.DIRECTION_TYPE direction, ElementStore store)
    {
        List<Association> associations = transformerModelElement.getAssociations();

        if (associations == null)
            return null;

        for (Association association : associations)
        {
            String stereotype = association.getProperties() == null ? null : association.getProperties().getStereotype();
            if (TransformerFactory.DIRECTION_TYPE.leftToRight.equals(direction))
            {
                if (isSourceAssociations && LEFT_STEREOTYPE_PROPERTY.equals(stereotype))
                {
                    return retrieveAttributeFromAssociation(association, store);
                }
                else if (!isSourceAssociations && RIGHT_STEREOTYPE_PROPERTY.equals(stereotype))
                {
                    return retrieveAttributeFromAssociation(association, store);
                }

            }
            else if (TransformerFactory.DIRECTION_TYPE.rightToLeft.equals(direction))
            {
                if (isSourceAssociations && RIGHT_STEREOTYPE_PROPERTY.equals(stereotype))
                {
                    return retrieveAttributeFromAssociation(association, store);
                }
                else if (!isSourceAssociations && LEFT_STEREOTYPE_PROPERTY.equals(stereotype))
                {
                    return retrieveAttributeFromAssociation(association, store);
                }
            }
        }

        return null;
    }

    protected ModelElement retrieveAttributeFromAssociation(Association association, ElementStore store)
    {
        if (association.getTargetAttributeId() != null)
        {
            return store.get(association.getTargetAttributeId());
        }
        else
        {
            return store.get(association.getTargetClassId());
        }
    }

    protected List<ModelElementMapper> retrieveMappersFromAssociations(ModelElement transformerModelElement, ElementStore store,
                                                                       TransformerFactory.DIRECTION_TYPE direction)
    {
        List<Association> associations = transformerModelElement.getAssociations();
        if (associations == null)
        {
            throw new UnexpectedException(String.format("Unable to perform transformation for transformer %s . " +
                    "No associated mappers found.", transformerModelElement.getGuid()));
        }


        List<ModelElementMapper> modelElementMappers = new ArrayList<ModelElementMapper>();
        for (Association association : associations)
        {
            String stereotype = association.getProperties() == null ? null : association.getProperties().getStereotype();
            String mapperDirection = association.getProperties() == null ? null : association.getProperties().getDirection();
            if (SOURCE_DESTINATION_DIRECTION_PROPERTY.equals(mapperDirection) && !association.isIsNavigable()
                    && !POST_PROCESSING_STEREOTYPE_PROPERTY.equals(stereotype))
            {
                String destinationMapperId = association.getTargetClassId();
                ModelElementMapper modelElementMapper = createMapper(destinationMapperId, store, direction);
                modelElementMapper.setStore(store);

                if (!(modelElementMapper instanceof PostProcessorMapper))
                {
                    modelElementMappers.add(modelElementMapper);
                }
            }
        }
        return modelElementMappers;
    }

    protected List<PostProcessorMapper> retrievePostProcessorMappersFromAssociations(ModelElement transformerModelElement, ElementStore store,
                                                           TransformerFactory.DIRECTION_TYPE direction)
    {
        List<Association> associations = transformerModelElement.getAssociations();
        if (associations == null)
        {
            throw new UnexpectedException(String.format("Unable to perform transformation for transformer %s . " +
                    "No associated mappers found.", transformerModelElement.getGuid()));
        }


        List<PostProcessorMapper> postProcessorMappers = new ArrayList<PostProcessorMapper>();
        for (Association association : associations)
        {
            String stereotype = association.getProperties() == null ? null : association.getProperties().getStereotype();
            String mapperDirection = association.getProperties() == null ? null : association.getProperties().getDirection();
            if (SOURCE_DESTINATION_DIRECTION_PROPERTY.equals(mapperDirection) && !association.isIsNavigable()
                    && POST_PROCESSING_STEREOTYPE_PROPERTY.equals(stereotype))
            {
                String destinationMapperId = association.getTargetClassId();
                Mapper mapper = createMapper(destinationMapperId, store, direction);
                if (mapper instanceof PostProcessorMapper)
                {
                    postProcessorMappers.add((PostProcessorMapper)mapper);
                }
            }
        }
        return postProcessorMappers;
    }

    protected abstract ModelElementMapper createMapper(String mapperGuid, ElementStore store,
                                                       TransformerFactory.DIRECTION_TYPE direction);

    public void setObjectTypeMapper(ObjectTypeMapper objectTypeMapper)
    {
        this.objectTypeMapper = objectTypeMapper;
    }


}
