package net.sf.javaprinciples.data.transformer;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.jcc.model.parser.UnexpectedException;
import net.sf.jcc.model.parser.uml2.ElementStore;
import net.sf.jcc.model.parser.uml2.ModelElement;

/**
 *  Factory to create a {@link MapperTransformer}
 *  from the list of registered transformers.
 *
 * @author Kay Chevalier
 */
public class MapperTransformerFactory<T,V> extends ModelElementFactory implements TransformerFactory
{
    private List<TransformerDetails> registeredTransformers;
    private ModelElementMapperFactory modelElementMapperFactory;
    private static final Logger log = LoggerFactory.getLogger(MapperTransformerFactory.class);

    @Override
    public Transformer<T,V> createTransformer(String transformerId, DIRECTION_TYPE direction)
    {
        log.info("Creating transformer for transformerId {} and direction {}", transformerId, direction);
        TransformerDetails transformerDetails = getTransformerDetails(transformerId);
        if (transformerDetails == null)
            return null;

        ElementStore store = transformerDetails.getStore();
        ModelElement transformerModelElement = store.get(transformerDetails.getTransformerId());

        MapperTransformer<T, V> mapperTransformer = new MapperTransformer<T, V>();

        mapperTransformer.setSourceModelElement(retrieveSourceFromAssociations(transformerModelElement, direction, store));
        mapperTransformer.setDestinationModelElement(retrieveDestinationFromAssociations(transformerModelElement, direction, store));

        mapperTransformer.setModelElementMappers(retrieveMappersFromAssociations(transformerModelElement, store, direction));
        mapperTransformer.setStore(store);
        mapperTransformer.setObjectTypeMapper(objectTypeMapper);
        return mapperTransformer;
    }

    @Override
    protected ModelElementMapper<T,V> createMapper(String mapperGuid, ElementStore store,
                                                   TransformerFactory.DIRECTION_TYPE direction)
    {
        return modelElementMapperFactory.createMapper(mapperGuid, store, direction);
    }


    private TransformerDetails getTransformerDetails(String transformerId)
    {
        if (registeredTransformers != null)
        {
            for (TransformerDetails transformerDetails : registeredTransformers)
            {
                if (transformerDetails.getTransformerId().equals(transformerId))
                    return transformerDetails;
            }
        }

        throw new UnexpectedException("No registered transformers exist for transformer id " + transformerId);
    }

    public void setRegisteredTransformers(List<TransformerDetails> registeredTransformers)
    {
        this.registeredTransformers = registeredTransformers;
    }

    public void setModelElementMapperFactory(ModelElementMapperFactory modelElementMapperFactory)
    {
        this.modelElementMapperFactory = modelElementMapperFactory;
    }
}
