package net.sf.javaprinciples.data.transformer;

import java.util.List;

import net.sf.jcc.model.parser.uml2.ElementStore;
import net.sf.jcc.model.parser.uml2.ModelElement;

/**
 * Transformer to transform and map from an input object to an output object.
 * Uses a source and destination model element to determine input and output
 * and model element mappers to delegate mapping functionality to.
 *
 * @author Kay Chevalier
 */
public class MapperTransformer<Input, Output> implements Transformer<Input, Output>
{
    private ElementStore store;
    private ModelElement sourceModelElement;
    private ModelElement destinationModelElement;
    private List<ModelElementMapper> modelElementMappers;
    private ObjectTypeMapper objectTypeMapper;

    public Output transform(Input input)
    {
        Output output = instantiateOutput(destinationModelElement);
        callMappers(input, output);

        return output;
    }

    private Output instantiateOutput(ModelElement destinationModelElement)
    {
        return (Output) ModelElementMapperHelper.createObjectFromClassModelElement(destinationModelElement, store, objectTypeMapper);
    }

    private void callMappers(Input input, Output output) throws AttributeNotFoundException
    {
        boolean attributesMapped = false;

        for (ModelElementMapper mapper : modelElementMappers)
        {
            try {
                mapper.map(input, output);
                attributesMapped = true;
            }
            catch (AttributeNotFoundException e)
            {

            }
        }

        if (!attributesMapped)
            throw new AttributeNotFoundException(String.format("No attributes mapped from input %s to output %s",
                    input.getClass().getName(), output.getClass().getName()));
    }

    public void setStore(ElementStore store)
    {
        this.store = store;
    }

    public void setSourceModelElement(ModelElement sourceModelElement)
    {
        this.sourceModelElement = sourceModelElement;
    }

    public void setDestinationModelElement(ModelElement destinationModelElement)
    {
        this.destinationModelElement = destinationModelElement;
    }

    public void setModelElementMappers(List<ModelElementMapper> modelElementMappers)
    {
        this.modelElementMappers = modelElementMappers;
    }
    public ModelElement getSourceModelElement()
    {
        return sourceModelElement;
    }

    public ModelElement getDestinationModelElement()
    {
        return destinationModelElement;
    }

    public List<ModelElementMapper> getModelElementMappers()
    {
        return modelElementMappers;
    }

    public void setObjectTypeMapper(ObjectTypeMapper objectTypeMapper)
    {
        this.objectTypeMapper = objectTypeMapper;
    }
}
