package net.sf.javaprinciples.data.transformer;

import java.util.Collection;

import net.sf.javaprinciples.core.ObjectClassHelper;

/**
 * Maps an input object to an output object.  Delegates attribute mapping
 * to its list of mappers.
 *
 * @author Kay Chevalier
 */
public class ComplexMapper<T, U> extends ModelElementMapper<T, U>
{

    @Override
    public void map(T input, U output)
    {
        input = retrieveInput(getSourceModelElement(), input);

        if (input == null)
        {
            throw new AttributeNotFoundException("Source Model Element input class is null - no attributes to set");
        }

        if (input instanceof Collection<?>)
        {
            for (Object inputObject : (Collection)input)
            {
                mapObject((T)inputObject, output);
            }
        }
        else
        {
            mapObject(input, output);
        }

        performPostProcessingMapping(input, output);
    }

    private void mapObject(T input, U output)
    {
        U destinationOutput = null;
        String attributeName = null;
        Collection<U> outputCollection = null;

        if(getDestinationModelElement() != null)
        {
            attributeName =  ModelElementMapperHelper.getPropertyNameFromElement(getDestinationModelElement());

            destinationOutput = (U)objectTypeMapper.getAttributeFromObject(output, attributeName);

            if (destinationOutput instanceof Collection<?>)
            {
                outputCollection = (Collection<U>)destinationOutput;
                String packageName = ModelElementMapperHelper.getPackageNameFromParentElement(getDestinationModelElement(), store);
                String propertyType = ModelElementMapperHelper.getAttributeTypeFromProperties(getDestinationModelElement());
                packageName = objectTypeMapper.formatPackageName(packageName);
                destinationOutput = (U)ObjectClassHelper.createObjectFromClassName(String.format("%s.%s", packageName, propertyType));
            }

            if (destinationOutput == null)
                destinationOutput = instantiateOutputFromAttributeName(output, attributeName);
        }

        boolean attributeSet = processMapperAssociations(input,
                destinationOutput == null ? output : destinationOutput);

        String sourceModelElementName = ModelElementMapperHelper.getPropertyNameFromElement(getSourceModelElement());
        String outputClassName = destinationOutput == null ? output.getClass().getName() : destinationOutput.getClass().getName();
        logAttributeMapping(input.getClass().getName(), sourceModelElementName, outputClassName, attributeName,
                destinationOutput == null ? output : destinationOutput);

        if (attributeSet)
        {
            if (destinationOutput != null)
            {
                if (outputCollection != null)
                {
                    outputCollection.add(destinationOutput);
                }
                else
                {
                    objectTypeMapper.assignAttributeToObject(destinationOutput, output, attributeName);
                }
            }
        }
        else
        {
            throw new AttributeNotFoundException("No attributes set by mapper associations for complex mapper");
        }
    }
}
