package net.sf.itcb.common.portlet.vaadin.interceptor.page;

import net.sf.itcb.common.portlet.vaadin.component.ItcbComponent;
import net.sf.itcb.common.portlet.vaadin.component.ItcbPage;
import net.sf.itcb.common.portlet.vaadin.component.ItcbPage.Updatable;
import net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor;
import net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor.ReloadOrder;
import net.sf.itcb.common.portlet.vaadin.page.impl.PageMappingProcessorImpl;

import com.vaadin.ui.UriFragmentUtility;
import com.vaadin.ui.UriFragmentUtility.FragmentChangedListener;

/**
 * This interface can be implemented to build your own interceptor which are called when the user change his page.
 * Those interceptors are called by :
 * <ul>
 * 	<li>{@link PageMappingProcessor#displayPage(String, ReloadOrder)}</li>
 *  <li>{@link FragmentChangedListener#fragmentChanged(com.vaadin.ui.UriFragmentUtility.FragmentChangedEvent) used by {@link UriFragmentUtility} in {@link PageMappingProcessorImpl}
 * <ul>
 * 	<li>handleBeforeChangePage method is called before changing page</li>
 *  <li>handleAfterChangePage method is called after the new page has been displayed when no exception has been thrown</li>
 *  <li>handleChangePageException is called when an exception has been thrown</li> 
 * </ul>
 * @since 0.5.0
 * @author Pierre Le Roux
 */
public interface ChangePageInterceptor {
	
	/**
	 * This enum contains possible values for effective executed actions during a page change call
	 * <ul>
	 *   <li>NONE : no action has been executed : page had been displayed with the same content as before</li>
	 *   <li>FULL_LOAD : page init method had been called and full content has been built for display</li>
	 *   <li>UPDATE : only  {@link Updatable#update()} method has been called if {@link ItcbPage} implements {@link Updatable}
	 * @author Pierre Le Roux
	 */
	public enum ExecutedAction {
		NONE,
		FULL_LOAD,
		UPDATE		
	}
	
	/**
	 * This method is called before executing change page methods.<br/>
	 * @param pageRef reference of the page to display
	 * @param itcbComponent the page or the controller linked to pageRef
	 * @param reloadOrder reload has been asked or not for this display
	 * @return An object if ChangePageInterceptor has to share vars that has to be keeped for handleAfterChangePage or handleAfterChangePageException 
	 */
	Object handleBeforeChangePage(String pageRef, ItcbComponent itcbComponent, ReloadOrder reloadOrder);
	
	/**
	 * This method is called after change page has been done when no exception had been thrown.
	 * @see ChangePageInterceptor#handleChangePageException when an exception is thrown
	 * @param pageRef reference of the page that has been displayed
	 * @param itcbComponent the page or the controller linked to pageRef
	 * @param reloadOrder reload had been asked or not for this display
	 * @param executedAction effective executed actions
	 * @param requestContextVar {@link ChangePageInterceptor#handleBeforeChangePage(String, ReloadOrder)} return object in which context variables can have been stored
	 */
	void handleAfterChangePage(String pageRef, ItcbComponent itcbComponent, ReloadOrder reloadOrder, ExecutedAction executedAction, Object requestContextVar);
	
	/**
	 * This method is called after change page when an exception had been thrown.
	 * @see ChangePageInterceptor#handleChangePageException when an exception is thrown
	 * @param pageRef reference of the page that has been displayed
	 * @param itcbComponent the page or the controller linked to pageRef
	 * @param reloadOrder reload had been asked or not for this display
	 * @param executedAction effective executed actions
	 * @param requestContextVar {@link ChangePageInterceptor#handleBeforeChangePage(String, ReloadOrder)} return object in which context variables can have been stored
	 * @param throwable thrown exception
	 */
	void handleChangePageException(String pageRef, ItcbComponent itcbComponent, ReloadOrder reloadOrder, ExecutedAction executedAction, Object requestContextVar, Throwable throwable);

}
