package net.sf.itcb.common.portlet.vaadin.exception;

import java.util.HashMap;
import java.util.Map;

import net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import com.vaadin.terminal.Terminal;


/**
 * The exception handler mapping stores a map in order to execute a specific mecanism for the exception.
 * The pageMappingProcessor to do something in pages when an error occured
 * @author Pierre Le Roux
 */
public class ExceptionHandlerMapping{
	
	private Map<Class, ExceptionHandler> exceptionMapping;
	
	protected final Logger log = LoggerFactory.getLogger(getClass());
	
	
	public ExceptionHandlerMapping() {
		this.exceptionMapping = new HashMap<Class, ExceptionHandler>();
	}	
	
	/**
	 * Set the exception mapping in a map :
	 * <ul>
	 * <li>key : the name of the exception class that has to be managed (or subclasses)</li>
	 * <li>value : the ExceptionHandler that will handle the exception</li>
	 * </ul>
	 * @param exceptionMapping
	 */
	@Required
	public void setExceptionMapping(Map<String, ExceptionHandler> exceptionMapping) {
		log.debug("Initializing exception handlers...");
		try {
			for (Map.Entry<String, ExceptionHandler> entry : exceptionMapping.entrySet()) {
				log.debug("{} Specific handler set", entry.getKey());
				this.exceptionMapping.put(Class.forName(entry.getKey()), entry.getValue());
			}
		}
		catch(ClassNotFoundException cnfe) {
			log.error("ExceptionHandlerMapping is not set because a class doesn't exist in classpath.", cnfe);
		}
	}

	/**
	 * Handle an exception according to the exception mapping set by Spring injection
	 * @param e the exception thrown that has to be handle
	 * @param pageMappingProcessor the user pageMappingProcessor in order to display exceptions...
	 */
	public void handleError(Terminal.ErrorEvent error, PageMappingProcessor pageMappingProcessor) {
		if(error != null) {
			Throwable t = error.getThrowable().getCause();
			if(t != null) {
				ExceptionHandler handler = exceptionMapping.get(t.getClass());
				if (handler == null) {
					for (Class entry : exceptionMapping.keySet()) {
						if( entry.isInstance(t) ) {
							handler = exceptionMapping.get(entry);
							break;
						}
					}
				}
				if(handler != null) {
					handler.handleError(error, pageMappingProcessor);
					return;
				}
			}
		}
		// If no exception handler for this exception, we only log it
		log.error("An unexpected error occurs : ", error.getThrowable().getMessage());
	}
}
