package net.sf.itcb.common.portlet.vaadin.page.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.portlet.PortletRequest;
import javax.portlet.PortletSession;
import javax.servlet.ServletRequest;

import net.sf.itcb.common.business.core.ItcbApplicationContextHolder;
import net.sf.itcb.common.portlet.exceptions.PortletItcbException;
import net.sf.itcb.common.portlet.exceptions.PortletItcbExceptionMappingErrors;
import net.sf.itcb.common.portlet.portal.InnerPortalAdapter;
import net.sf.itcb.common.portlet.vaadin.ItcbVaadinApplication;
import net.sf.itcb.common.portlet.vaadin.component.ItcbComponent;
import net.sf.itcb.common.portlet.vaadin.component.ItcbController;
import net.sf.itcb.common.portlet.vaadin.component.ItcbPage;
import net.sf.itcb.common.portlet.vaadin.component.ItcbPage.FieldsValuePreLoadable;
import net.sf.itcb.common.portlet.vaadin.interceptor.page.ChangePageInterceptor;
import net.sf.itcb.common.portlet.vaadin.interceptor.page.ChangePageInterceptor.ExecutedAction;
import net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor;
import net.sf.itcb.common.portlet.vaadin.page.config.PageMappingProcessorConfig;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;

import com.vaadin.data.Validatable;
import com.vaadin.terminal.ParameterHandler;
import com.vaadin.terminal.Terminal.ErrorEvent;
import com.vaadin.terminal.URIHandler;
import com.vaadin.terminal.VariableOwner;
import com.vaadin.terminal.gwt.server.ChangeVariablesErrorEvent;
import com.vaadin.ui.ComponentContainer;
import com.vaadin.ui.ProgressIndicator;
import com.vaadin.ui.UriFragmentUtility;
import com.vaadin.ui.UriFragmentUtility.FragmentChangedEvent;
import com.vaadin.ui.UriFragmentUtility.FragmentChangedListener;

/**
 * ITCB implementation of PageMappingProcessor
 * 
 * @author Pierre Le Roux
 *
 */
public class PageMappingProcessorImpl implements PageMappingProcessor, Serializable {

	private static final long serialVersionUID = 1L;
	
	protected final Logger log = LoggerFactory.getLogger(getClass());
	

	ConcurrentMap<String, ItcbComponent> mapping;
	
	protected String defaultPageKey;
	
	PageMappingProcessorConfig pageMappingProcessorConfig;
	
	protected ComponentContainer contentContainer;
	
	protected PortletRequest portletInitRequest;
	
	protected ServletRequest servletInitRequest;	
	
	protected ItcbVaadinApplication application;
	
	protected ItcbPage currentPage;
	
	protected PageMappingProcessorAsyncDelegate asyncDelegate;
	
	protected UriFragmentUtility uriFragmentUtility;
	
	ProgressIndicator progressIndicator;
	
	protected String firstPageKey;
		
	/**
	 * By default, the default page is displayed at portlet initialization<br/>
	 * If the portlet doesn't want to use this processor at portlet initialization,
	 * Spring context has to be set to false in Spring configuration
	 */
	protected Boolean isAutomaticDisplay = true;

	private String currentPageRef;

	private boolean refreshApplication=true;
	
	private InnerPortalAdapter portalAdapter;
	
	public PageMappingProcessorImpl() {
		
	}
	
// Getters / setters
	
	public Map<String, ItcbComponent> getMapping() {
		return mapping;
	}

	/**
	 * Defines the components mapping : pages or controllers
	 * This function has to be used if you want to load all the pages at application initialization.<br/>
	 * If your application contains too many pages, you should inject a map with key and ItcbComponent bean name as value {@link PageMappingProcessorConfig#setMappingReferences(Map)}
	 * @param mapping mapping that will be used to reach a page
	 */
	public void setMapping(Map<String, ItcbComponent> mapping) {
		this.mapping = new ConcurrentHashMap<String, ItcbComponent>();
		this.mapping.putAll(mapping);
	}

	public String getDefaultPageKey() {
		return defaultPageKey;
	}

	/**
	 * Define the default page. This key has to be in mapping list
	 * Has to be set by Spring context at creation
	 * @param defaultPage the default page to show
	 */
	@Required
	public void setDefaultPageKey(String defaultPageKey) {
		this.defaultPageKey = defaultPageKey;
	}
	
	
	public void setPageMappingProcessorConfig(PageMappingProcessorConfig pageMappingProcessorConfig) {
		this.pageMappingProcessorConfig = pageMappingProcessorConfig;
	}
	
	public PageMappingProcessorConfig getPageMappingProcessorConfig() {
		return pageMappingProcessorConfig;
	}
	
	// Application Specific params
	
	@Override
	public void setInitRequest(PortletRequest portletRequest) {
		this.portletInitRequest = portletRequest;
	}
	
	@Override
	public void setInitRequest(ServletRequest servletRequest) {
		this.servletInitRequest=servletRequest;
		ComponentContainer layout =initLayout();
		if(layout != null) {
			application.getMainWindow().setContent(layout);
		}
		if(mapping == null) {
			this.mapping = new ConcurrentHashMap<String, ItcbComponent>();
		}
		contentContainer = getContentContainer();
	}
	
	/**
	 * Nothing in this class to initialize the layout. Developers can override this method such as {@link AbstractCustomLayoutPageMappingProcessorImpl} does
	 */
	protected ComponentContainer initLayout() {
		return null;
	}

	/**
	 * Get the layout of your application. It can be useful if you want to define a custom layout for your application.
	 * If this method is implemented specifically, you also have to implement initLayout and getPanel with the panel in which your pages are going to change
	 */
	protected ComponentContainer getLayout() {
		// No specific layout in basic implementation
		return application.getMainWindow();
	}

	/**
	 * Get the panel in which pages will be displayed. 
	 * In most cases the panel is the main window but it can be different if you want to define a left column, an header or a footer.
	 * @see AbstractCustomLayoutPageMappingProcessorImpl
	 * @see PageMappingProcessor#getLayout()
	 * @param componentContainer componentContainer to be set by the application : 
	 */
	protected ComponentContainer getContentContainer() {
		// Basic implementation in which pages are changed in main window
		return application.getMainWindow();
	}
	
	@Override
	public void setApplication(ItcbVaadinApplication application) {
		this.application = application;
	}

	@Override
	public ItcbVaadinApplication getApplication() {
		return application;
	}
	
	// Functions
	
	@Override
	public void displayPage(String refPage, ReloadOrder reload) {
		log.debug("displayPage {} {}" , refPage, reload);
			if(reload != null) {
				changePage(refPage, reload);
				uriFragmentUtility.setFragment(refPage, false);
				  if(firstPageKey == null) {
						firstPageKey = refPage;
				  }
			}		
	}
	
	@Override
	public void displayDefaultPage() throws Exception {
		log.debug("Display the default page");
		if(defaultPageKey != null) {
			uriFragmentUtility = new UriFragmentUtility();
			uriFragmentUtility.addListener(new FragmentChangedListener() {
				   /**
				 * 
				 */
				private static final long serialVersionUID = 1L;

				public void fragmentChanged(FragmentChangedEvent source) {
				      if(currentPage instanceof Validatable) {
				    	  try  {
				    		  ((Validatable)currentPage).validate();
				    	  }
				    	  catch(Exception e) {
				    		  uriFragmentUtility.setFragment(currentPageRef, false);
				    		  handleException(e);
				    		  return;
				    	  }
				      }
					  String fragment = source.getUriFragmentUtility().getFragment();
				      if (fragment != null) {
					  	  log.debug("changeFragment {}", fragment);
						  String refPage="";
				    	  String[] fragmentParts = fragment.split("-");
					      refPage=fragmentParts[0];
					      
						  if(refPage.equals("")) {
							  if(refreshApplication == true) {
								refPage = currentPageRef;
							  }
							  else {
								  refPage = firstPageKey;
							  }
							  log.debug("No fragment on call so calling {}", refPage);
						  }
						  if(!refPage.equals(currentPageRef)) {
							  if(!pageMappingProcessorConfig.isAuthorizeDirectAccessToFragmentOnLoad() && !isAlreadyLoaded(refPage)) {
								log.debug("Direct access to {} not allowed", refPage);
								refPage = currentPageRef;
								uriFragmentUtility.setFragment(refPage, false);
								log.debug("Calling {} instead", refPage);
							  }
								
					    	  changePage(refPage, ReloadOrder.FALSE);
					    	  refreshApplication=false;
					      }
				      }
				   }

				private boolean isAlreadyLoaded(String refPage) {
					ItcbComponent component = mapping.get(refPage);
					if(component != null && component instanceof ItcbPage && ((ItcbPage)component).getParent() != null) {
						return true;
					}
					return false;
				}
				});
			contentContainer.addComponent(uriFragmentUtility);
			ItcbComponent component=  mapping.get(defaultPageKey);
			if(component == null) {
				synchronized (mapping) {
					component = ItcbApplicationContextHolder.getContext().getBean(pageMappingProcessorConfig.getMappingReferences().get(defaultPageKey), ItcbComponent.class);
					mapping.put(defaultPageKey, component);
				}
			}
			if(component instanceof ItcbController) {
				component.setPageMappingProcessor(this);
				component.init();
			}
			else {
				displayPage(defaultPageKey, ReloadOrder.TRUE);
			}
		}
		else {
			if(log.isDebugEnabled()) {
				log.warn("No default page configured. PageMappingProcessor mecanism not used. Set the defaultPageKey property in your spring context.");
			}
		}
	}
	
	@Override
	public void displayPreviousPage() {
		contentContainer.getApplication().getMainWindow().executeJavaScript("history.back()");
	}
	
	@Override
	public void setSessionAttribute (String key, Object value) {
		if(value == null) {
			application.getApplicationSession().remove(key);
		}
		else {
			application.getApplicationSession().put(key, value);
		}
	}
	
	@Override
	@SuppressWarnings("unchecked")
	public <Y> Y getSessionAttribute(String key, Class<Y> requiredType) {
		return (Y)application.getApplicationSession().get(key );
	}
	
	@Override
	@SuppressWarnings("unchecked")
	public <Y> Y getSharedSessionAttribute(String key, Class<Y> requiredType) {
		if(portletInitRequest != null) {
			return (Y) portletInitRequest.getPortletSession().getAttribute(key, PortletSession.APPLICATION_SCOPE);
		}
		return (Y)servletInitRequest.getAttribute(key);
	}
	
	@Override
	public void setSharedSessionAttribute(String key, Object value) {
		if(portletInitRequest != null) {
			portletInitRequest.getPortletSession().setAttribute(key, value, PortletSession.APPLICATION_SCOPE);
		}
		else {
			servletInitRequest.setAttribute(key, value);
		}
	}
	
	@Override
	public Object getRequest() {
		if(portletInitRequest != null) {
			return portletInitRequest;
		}
		else {
			return servletInitRequest;
		}
	}

	@Override
	public boolean isAutomaticDisplay() {
		return isAutomaticDisplay;
	}
	
	@Override
	public void handleException(Throwable e) {
		if(e instanceof RuntimeException) {
			throw (RuntimeException)e;
		}
		else {
			MessageSource itcbPortletMessageSource = ItcbApplicationContextHolder.getContext().getBean("common-portletResources", MessageSource.class);
			throw new PortletItcbException(PortletItcbExceptionMappingErrors.COMMON_PORTLET_LOADING_PAGE, itcbPortletMessageSource.getMessage("common.portlet.exception.loading.page", new Object[] {e.getMessage()}, LocaleContextHolder.getLocale()));
		}
	}
	
	@Override
	public void handleError(ErrorEvent error) {
		try {
			pageMappingProcessorConfig.getExceptionHandlerMapping().handleError(error, this);
		}
		catch(Exception exceptionInExceptionManagement) {
			log.error("The exception has not been handled well {}. This Exception occured {}", error.getThrowable(), exceptionInExceptionManagement);
		}
	}

	@Override
	public String getRequestParameter(String key) {
		String value = null;
		if(portletInitRequest !=null) {
		    value = portletInitRequest.getParameter(key);
		}
		if(value == null) {
			value = servletInitRequest.getParameter(key);
		}
		return value;
	}

	@Override
	public Object getErrorOwner(ErrorEvent event) {
		Object owner = null;
        if (event instanceof VariableOwner.ErrorEvent) {
            owner = ((VariableOwner.ErrorEvent) event).getVariableOwner();
        } else if (event instanceof URIHandler.ErrorEvent) {
            owner = ((URIHandler.ErrorEvent) event).getURIHandler();
        } else if (event instanceof ParameterHandler.ErrorEvent) {
            owner = ((ParameterHandler.ErrorEvent) event).getParameterHandler();
        } else if (event instanceof ChangeVariablesErrorEvent) {
            owner = ((ChangeVariablesErrorEvent) event).getComponent();
        }

        return owner;
		
	}

	@SuppressWarnings("unchecked")
	@Override
	public <Z extends ItcbPage> Z getPage(String pageRef,
			Class<Z> requiredType) {
		
		ItcbComponent component = mapping.get(pageRef);
		if(component == null) {
			synchronized (mapping) {
				component = ItcbApplicationContextHolder.getContext().getBean(pageMappingProcessorConfig.getMappingReferences().get(pageRef), ItcbComponent.class);
				mapping.put(pageRef, component);
			}
		}
		
		if(component instanceof ItcbPage) {
			return (Z) component;
		}
		else {
			MessageSource itcbPortletMessageSource = ItcbApplicationContextHolder.getContext().getBean("common-portletResources", MessageSource.class);
			throw new PortletItcbException(PortletItcbExceptionMappingErrors.COMMON_PORTLET_TYPE_ERROR_CONTROLLER, itcbPortletMessageSource.getMessage("common.portlet.exception.type.error.controller", new Object[] {pageRef}, LocaleContextHolder.getLocale()));
		}
	}

	@Override
	/**
	 * This function is called with async method in order to construct the page serverside without time issue for the user
	 */
	public void preloadPage(String refPage, ReloadOrder reload) {
		if(progressIndicator==null) {
			progressIndicator = new ProgressIndicator();
			progressIndicator.setIndeterminate(true);
			progressIndicator.setPollingInterval(5000);
			//progressIndicator.setVisible(false);
			progressIndicator.addStyleName(PageMappingProcessorAsyncDelegate.HIDDEN_STYLE);
	        contentContainer.addComponent(progressIndicator);
		}
		if(asyncDelegate == null) {
			asyncDelegate= new PageMappingProcessorAsyncDelegate();
		}
		asyncDelegate.preloadPage(refPage, reload, this);
		progressIndicator.setEnabled(true);
		
	}
	
	/**
	 * This method is called in order to change page and to display it
	 * @param pageRef
	 * @param reloadOrder
	 */
	protected void changePage(String pageRef, ReloadOrder reloadOrder) {
        log.debug("changePage {} {}" , pageRef, reloadOrder);
        ItcbComponent component = mapping.get(pageRef);
        if(component == null) {
            synchronized (mapping) {
                component = ItcbApplicationContextHolder.getContext().getBean(pageMappingProcessorConfig.getMappingReferences().get(pageRef), ItcbComponent.class);
                mapping.put(pageRef, component);
            }
        }
       
        if(component == null) {
            MessageSource itcbPortletMessageSource = ItcbApplicationContextHolder.getContext().getBean("common-portletResources", MessageSource.class);
            throw new PortletItcbException(PortletItcbExceptionMappingErrors.COMMON_PORTLET_MISSING_PAGE, itcbPortletMessageSource.getMessage("common.portlet.exception.missing.page", new Object[] {pageRef}, LocaleContextHolder.getLocale()));
        }
        List<ChangePageInterceptor> changePageInterceptors = pageMappingProcessorConfig.getChangePageInterceptors();
        List<Object> listRequestContext = new ArrayList<Object>();
        ExecutedAction executedAction = ExecutedAction.NONE;
        try {
            for (ChangePageInterceptor changePageInterceptor : changePageInterceptors) {
                listRequestContext.add(changePageInterceptor.handleBeforeChangePage(pageRef, component, reloadOrder));
            }
            if( component instanceof ItcbPage) {
                //ItcbPage currentPageTemp=currentPage;
                String currentPageRefTemp=currentPageRef;
               
                ItcbPage page = (ItcbPage)component;
                if(currentPage != null && currentPage != component) {
                    currentPage.setVisible(false);
                }
               
                if (page.getParent() == null || reloadOrder.equals(ReloadOrder.TRUE) ) {
                    loadPage(page, reloadOrder);
                    executedAction = ExecutedAction.FULL_LOAD;
                }
                // If the page has already been loaded and the page has a method to call to be updated each time and reloadOrder was FALSE
                // then we update the page according to Updatable.update method
                else if (page instanceof ItcbPage.Updatable) {
                    ((ItcbPage.Updatable)page).update();
                    executedAction = ExecutedAction.UPDATE;
                }
                // If loadPage or updatePage ordered another page change then we don't display the pageRef related page. 
                // The other page display has already been ordered in process
                if(currentPageRef == null || (currentPageRefTemp != null && currentPageRefTemp.equals(currentPageRef))) {
                    page.setVisible(true);
                   
                    // If the page has already been preloaded, we remove the hidden style
                    page.removeStyleName(PageMappingProcessorAsyncDelegate.HIDDEN_STYLE);
                    currentPage=page;
                    currentPageRef=pageRef;
                }
            }
            else if( component instanceof ItcbController) {
                // TODO what have we to do when changePage is called for a controller in an application ?
            }
            for (int i = changePageInterceptors.size(); i > 0; i--) {
                changePageInterceptors.get(i-1).handleAfterChangePage(pageRef, component, reloadOrder, executedAction, listRequestContext.get(i-1));
            }
       
        }
        catch(Exception e) {
            RuntimeException re;
            if(! (e instanceof RuntimeException) ) {
                MessageSource itcbPortletMessageSource = ItcbApplicationContextHolder.getContext().getBean("common-portletResources", MessageSource.class);
                re = new PortletItcbException(PortletItcbExceptionMappingErrors.COMMON_PORTLET_LOADING_PAGE, itcbPortletMessageSource.getMessage("common.portlet.exception.loading.page", new Object[] {pageRef, e.getMessage()}, LocaleContextHolder.getLocale()), e);
            }
            else {
                re = (RuntimeException) e;
            }
            for (int i = changePageInterceptors.size(); i > 0; i--) {
                changePageInterceptors.get(i-1).handleChangePageException(pageRef, component, reloadOrder, executedAction, listRequestContext.get(i-1), e);
            }
            ((ItcbPage)component).setParent(null); // We reinitialize parent to null in order to force page loading even if reloadOrder is FALSE.
            contentContainer.removeComponent((ItcbPage)component);
            throw re;
        }

       
    }
	
	/**
	 * Load the page or reload it and attach it to the main panel (window)
	 * This function can be called either by {@link PageMappingProcessor#displayPage(String, net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor.ReloadOrder)} or {@link PageMappingProcessor#preloadPage(String, net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor.ReloadOrder)}
	 * @param page
	 * @param reload
	 * @throws Exception
	 */
	protected void loadPage(ItcbPage page, ReloadOrder reload)
			throws Exception {
		log.debug("loadPage {} {}" + page, reload);
		if(page != null && 
				(reload == ReloadOrder.TRUE)) {
			page.setPageMappingProcessor(this);
			page.removeAllComponents();
			if(page.getParent() == null) {
				contentContainer.addComponent(page);
			}
			page.init();
			if(page instanceof ItcbPage.FieldsValuePreLoadable) {
				((FieldsValuePreLoadable)page).fillFieldsWithValues();
			}
			
		}
		if(page.getPageMappingProcessor() == null || page.getParent() == null || !page.getComponentIterator().hasNext() ) {
			page.setPageMappingProcessor(this);
			contentContainer.addComponent(page);
			page.init();
			if(page instanceof ItcbPage.FieldsValuePreLoadable) {
				((FieldsValuePreLoadable)page).fillFieldsWithValues();
			}
		}
	}


	@Override
	public void refreshApplication() {
		this.refreshApplication= true;	
	}
	
	@Override
	public InnerPortalAdapter getPortalAdapter() {
		if(portalAdapter == null) {
			this.portalAdapter= new InnerPortalAdapter(getApplication().getApplicationConfig().getPortalAdapterProvider().getPortalAdapter(getRequest()), getRequest());
		}
		return portalAdapter;		
	}	
	
	public class PageMappingProcessorAsyncDelegate {
		
		public static final String HIDDEN_STYLE = "ITCB_HIDDEN_PRELOAD";
		protected final Logger log = LoggerFactory.getLogger(getClass());
		
		/**
		 * This function is called with async method in order to construct the page serverside without time issue for the user
		 * @param refPage the page to load
		 * @param reload reload or not or only if modified
		 * @param pageMappingProcessor application pageMappingProcessor
		 */
		public void preloadPage(final String refPage, final ReloadOrder reload, PageMappingProcessor pageMappingProcessor) {
				synchronized(PageMappingProcessorImpl.this) {
					log.debug("Preload {}", refPage);		
					Thread thread = new Thread() {
						@Override
				        public void run() {
							try  {
								ItcbComponent component = (ItcbComponent)mapping.get(refPage);
							
								if(component == null) {
									MessageSource itcbPortletMessageSource = ItcbApplicationContextHolder.getContext().getBean("common-portletResources", MessageSource.class);
									throw new PortletItcbException(PortletItcbExceptionMappingErrors.COMMON_PORTLET_MISSING_PAGE, itcbPortletMessageSource.getMessage("common.portlet.exception.missing.page", new Object[] {refPage}, LocaleContextHolder.getLocale()));
								}
								
								if( !(component instanceof ItcbPage) ) {
									MessageSource itcbPortletMessageSource = ItcbApplicationContextHolder.getContext().getBean("common-portletResources", MessageSource.class);
									throw new PortletItcbException(PortletItcbExceptionMappingErrors.COMMON_PORTLET_TYPE_ERROR_CONTROLLER, itcbPortletMessageSource.getMessage("common.portlet.exception.missing.page", new Object[] {refPage}, LocaleContextHolder.getLocale()));
								}
								
								ItcbPage page = (ItcbPage)component;
								loadPage(page, reload);	
								page.setVisible(true);
								page.addStyleName(HIDDEN_STYLE);
								progressIndicator.setEnabled(false);
							}
							catch(final Exception e) {
								log.error(e.getMessage());
							}
						}
					};
					thread.start();
					
				}
			
		}
		
		
		
		
	}
	
	
	
	
	
}
