package net.sf.itcb.common.portlet.vaadin.page;


import javax.portlet.PortletRequest;
import javax.servlet.ServletRequest;

import net.sf.itcb.common.portlet.portal.InnerPortalAdapter;
import net.sf.itcb.common.portlet.vaadin.ItcbVaadinApplication;
import net.sf.itcb.common.portlet.vaadin.component.ItcbPage;
import net.sf.itcb.common.portlet.vaadin.exception.ExceptionHandler;
import net.sf.itcb.common.portlet.vaadin.page.impl.PageMappingProcessorImpl;

import com.vaadin.terminal.Terminal.ErrorEvent;

/** 
 * PageMappingProcessor is the main controller for pages of the application.<br/>
 * The PageMappingProcessor's aim is to provide developers the toolkit they need to code Vaadin pages and make them interact with each others.<br/>
 * Some properties have to be set in Spring context : 
 * <ul>
	<li><code>pageMappingProcessorConfig</code> : global configuration for this pageMappingProcessor</li>
	<li><code>mapping</code> : page mapping with a key and an ItcbComponent as value</li>
	<li><code>defaultPageKey</code> : key in mapping the application has to use to load the first page of the application</li>
 * </ul>
 * Some others are set by {@link ItcbVaadinApplication} : 
 * <ul>
 *   <li><code>panel</code></li>
 *   <li><code>initRequest</code> (portlet or servlet depending on context)</li>
 *   <li><code>application</code></li>
 * </ul>
 *
 * Developers can also use session storage (portal and portlet scope) and request management (portlet and servlet) 
 * without having to deel with portals or session heterogeneity.<br/>
 * Developers can declare global exception management using <code>exceptionHandlerMapping</code> in <code>pageMappingProcessorConfig</code>.
 * 
 * @see PageMappingProcessorImpl 
 * 
 * @author Pierre Le Roux
 */
public interface PageMappingProcessor {

	/**
	 * enum used with displayPage methods for asking full reload of the page or not. 
	 * Could have others values in the future that's why we don't use simple boolean.
	 * @author Pierre Le Roux
	 */
	public enum ReloadOrder {
		TRUE,
		FALSE
	}	
	
	// Application Specific params
	
	/**
	 * The request is set by the application at the initialization. It allows to deal with session and to get servlet request parameters.
	 * @param request
	 */
	public void setInitRequest(PortletRequest request);
	
	/**
	 * The request is set by the application at the initialization. It is used when user manage vaadin webapplication instead of portlet application
	 * @param request
	 */
	public void setInitRequest(ServletRequest request);
	
	/**
	 * Set the current application object
	 * @param application
	 */
	public void setApplication(ItcbVaadinApplication application);

	/**
	 * Gets the application
	 * @return
	 */
	public ItcbVaadinApplication getApplication();
		
	/**
	 * Display the page associated with the id refPage. The page is simply displayed as it was the last time it has been displayed for this session
	 * If it is the first time, the page is initialized.
	 * If the <code>reload</code> parameter is set to <code>true</code>, page content is reloaded 
	 * @param refPage the page id referenced in <code>mapping</code>
	 * @param reload defines if the page has to be reloaded or not or only if modified
	 */
	public void displayPage(String refPage, ReloadOrder reload);
			
	/**
	 * Display the previous page
	 */
	public void displayPreviousPage();
	
	/**
	 * Display the default page defined in Spring context for the mapping processor
	 */
	public void displayDefaultPage() throws Exception;
	
	/**
	 * Store an attribute in session
	 * @param key
	 * @param value
	 */
	public void setSessionAttribute (String key, Object value);
	
	/**
	 * Get an object from his key in session 
	 * @param key
	 * @param requiredType
	 * @return 
	 */
	public <Y> Y getSessionAttribute(String key, Class<Y> requiredType);
	
	/**
	 * Get an object from his key in session 
	 * @param key
	 * @param requiredType
	 * @return 
	 */
	public <Y> Y getSharedSessionAttribute(String key, Class<Y> requiredType);
	
	/**
	 * Set a shared session attribute (the key must be auhorized to shared in portal session in portal properties).
	 * For example, in Liferay, it can be configured in portal-ext.properties by addind the key in session.shared.attributes
	 * @param key
	 * @param value
	 */
	public void setSharedSessionAttribute(String key, Object value);
	
	/**
	 * Return the portlet or the servlet request if needed for preferences, attributes...
	 */
	public Object getRequest();
	
	/**
	 * Get the parameter value in PortletRequest or in ServletRequest if not found in PortletRequest
	 * @param key
	 * @return value
	 */
	public String getRequestParameter(String key);

	/**
	 * Defines if the pageMappingProcessor is used when the context (view, edit, help) is launched or if it is used for displaying pages with a manual launch
	 * by calling displayDefaultPage
	 * @return
	 */
	public boolean isAutomaticDisplay();
	
	/**
	 * Handle an exception that occured in pages.
	 * @see ExceptionHandler
	 * @param e the exception
	 */
	public void handleException(Throwable e);
	
	/**
	 * Handle an error according to the mapping defined in Spring
	 * @param error
	 */
	public void handleError(ErrorEvent error);
	
	/**
	 * Get the owner of an ErrorEvent depending on its type
	 * @param error
	 */
	public Object getErrorOwner(ErrorEvent event);
	
	/**
	 * Retrieve a page already loaded from the page id in the PageMappingProcessor.
	 * @param <Z> type of page class
	 * @param refPage the id of the page in the PageMappingProcessor
	 * @param requiredType the type of page class
	 * @return the page with the specified type or null if the page has not been loaded yet
	 */
	public <Z extends ItcbPage> Z getPage(String refPage, Class<Z> requiredType);
	
	/**
	 * This function allow to preload a page without displaying it for now.<br/>
	 * It can be usefull when you want to have good display time and 
	 * if your next pages content is not linked to current page content and user actions.<br/>
	 * Parameters are set as in {@link PageMappingProcessor#displayPage(String, ReloadOrder) 
	 * @param refPage 
	 * @param reload
	 */
	public void preloadPage(String refPage, ReloadOrder reload);


	public void refreshApplication();	
	
	/**
	 * This function returns the portalAdapter according to the current request type
	 * @return PortalAdapter
	 */
	public InnerPortalAdapter getPortalAdapter();
	
	
}
