package net.sf.itcb.common.portlet.vaadin;

import java.net.SocketException;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.portlet.PortletMode;
import javax.portlet.PortletRequest;
import javax.portlet.PortletResponse;
import javax.portlet.RenderRequest;
import javax.portlet.ResourceRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.sf.itcb.common.business.core.ItcbApplicationContextHolder;
import net.sf.itcb.common.portlet.exceptions.PortletItcbException;
import net.sf.itcb.common.portlet.exceptions.PortletItcbExceptionMappingErrors;
import net.sf.itcb.common.portlet.vaadin.config.ItcbVaadinApplicationConfig;
import net.sf.itcb.common.portlet.vaadin.interceptor.request.ItcbRequestInterceptor;
import net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.Application;
import com.vaadin.terminal.ErrorMessage;
import com.vaadin.terminal.ParameterHandler;
import com.vaadin.terminal.SystemError;
import com.vaadin.terminal.Terminal;
import com.vaadin.terminal.URIHandler;
import com.vaadin.terminal.VariableOwner;
import com.vaadin.terminal.gwt.server.ChangeVariablesErrorEvent;
import com.vaadin.terminal.gwt.server.HttpServletRequestListener;
import com.vaadin.terminal.gwt.server.PortletRequestListener;
import com.vaadin.ui.AbstractComponent;
import com.vaadin.ui.Button;
import com.vaadin.ui.Window;



/**
 * This class is the vaadin base application in ITCB
 * It is configure by a spring bean that has to be named <code>itcbVaadinApplicationConfig</code>. 
 * This configuration bean has to be an implementation of {@link ItcbVaadinApplicationConfig}.<br/>
 * ItcbVaadinApplication assumes that the developer doesn't have (shouldn't have) to override this application 
 * unless the developer wants to override window name or vaadin error management (communication error, out of sync...).
 * This application can be used in Portlet mode or in Servlet mode. Both work.
 * 
 * <a href="http://itcommonbase.org:8080/confluence/display/ARCHI/Portals+and+Servlet+compatibility">Portals and Servlet compatibility</a>
 * 
 * @author Pierre Le Roux
 * @since 0.5.0
 */
public class ItcbVaadinApplication extends Application implements HttpServletRequestListener, PortletRequestListener, ItcbApplication {
	
	private static final String ITCB_VAADIN_APPLICATION_CONFIG_SPRING_KEY = "itcbVaadinApplicationConfig";

	private static final long serialVersionUID = 1L;
	
	protected final Logger log = LoggerFactory.getLogger(getClass());
	
	/**
	 * the main window in which {@link PageMappingProcessor} will work
	 */
	protected Window mainWindow;
	
	/**
	 * Application configuration : which {@link ItcbRequestInterceptor} have to be used on request, what are available {@link PageMappingProcessor} ...
	 */
	protected ItcbVaadinApplicationConfig itcbVaadinApplicationConfig;
		
	/**
	 * The current {@link PageMappingProcessor} used by application
	 */
	protected PageMappingProcessor currentPageMappingProcessor;
	
	/**
	 * The current mode key. It is linked to currentPageMappingProcessor.
	 */
	protected String currentMode;
	
	
	/** 
	 * Application session. It is needed because portlet sessions are kept even if application is reloaded. It means that old portlet session is the same for new application in the same session.
	 It shouldn't.
	 With this session, vars are stored for the application life time. If application is reloaded, applicationSession is renewed
	*/
	private ConcurrentMap<String, Object> applicationSession;
	
		
	/**
	 * Common method called on init request.<br/>
	 * Spring configuration is loaded such as the PageMappingProcessor associated to the {@link PortletMode} (in portal context)
	 * or to the <code>itcbmode</code> request parameter (in servlet context)
	 * @param servletRequest the servlet request when application is launched as web application
	 * @param portletRequest the portlet request when application is launched as portlet
	 * @param itcbmode the mode for which we want to initialize the associated {@link PageMappingProcessor} and display the first page
	 */
	protected void initItcb(HttpServletRequest servletRequest, PortletRequest portletRequest, String itcbmode) {
		log.debug("Enter initItcb");
		applicationSession = new ConcurrentHashMap<String, Object>();
		mainWindow = new Window(getWindowName());
		setMainWindow(mainWindow);
		
		if(itcbVaadinApplicationConfig == null) {
			itcbVaadinApplicationConfig= ItcbApplicationContextHolder.getContext().getBean(ITCB_VAADIN_APPLICATION_CONFIG_SPRING_KEY, ItcbVaadinApplicationConfig.class);
		}
		log.debug("Spring application config loaded");
		
		String pageMappingProcessorRef = itcbVaadinApplicationConfig.getMapPageMappingProcessor().get(itcbmode);
		if(pageMappingProcessorRef == null) {
			throw new PortletItcbException(PortletItcbExceptionMappingErrors.COMMON_PORTLET_MISSING_PMP, "the "+ itcbmode +" associated mapping variable has to be set in spring context. The bean name has to be : " + itcbmode + "ModePageMappingProcessor");
		}
	    PageMappingProcessor pmp = ItcbApplicationContextHolder.getContext().getBean(pageMappingProcessorRef, PageMappingProcessor.class);
		log.debug("PageMappingProcessor loaded");
		
		currentPageMappingProcessor = pmp;		
		currentMode=itcbmode;
		currentPageMappingProcessor.setApplication(ItcbVaadinApplication.this);
		
		if(servletRequest != null) {
			pmp.setInitRequest(servletRequest);
			if(itcbVaadinApplicationConfig.getItcbInitRequestInterceptors() != null) {
				for (ItcbRequestInterceptor requestFilter : itcbVaadinApplicationConfig.getItcbInitRequestInterceptors()) {
					requestFilter.handleRequestStart(this, servletRequest);
				}
			}
		}
		else {
			pmp.setInitRequest(portletRequest);
			pmp.setInitRequest(itcbVaadinApplicationConfig.getPortalAdapterProvider().getPortalAdapter(portletRequest).getOriginalServletRequest(portletRequest));

			if(itcbVaadinApplicationConfig.getItcbInitRequestInterceptors() != null) {
				for (ItcbRequestInterceptor requestFilter : itcbVaadinApplicationConfig.getItcbInitRequestInterceptors()) {
					requestFilter.handleRequestStart(this, portletRequest);
				}
			}
		}
		
		try {
			if(currentPageMappingProcessor.isAutomaticDisplay()) {
				currentPageMappingProcessor.displayDefaultPage();
			}
		} 
		catch (Exception e) {
				PortletItcbException portletItcbException = new PortletItcbException(PortletItcbExceptionMappingErrors.COMMON_PORTLET_LOADING_PAGE, e.getMessage(), e);
				Terminal.ErrorEvent errorEvent = new ChangeVariablesErrorEvent(mainWindow, portletItcbException, null);
				currentPageMappingProcessor.handleError(errorEvent);
				currentMode = null; // Force reload
		}
		log.debug("End initItcb");
	}

	@Override
	public void init() {
		// Nothing
	}
	
	/**
	 * terminalError does the same as superclass but also delegates error handling to  
	 * the current {@link PageMappingProcessor} exception handler.
	 * It allows to have the same behavior in all the application when a same kind of Exception is thrown
	 */
	@Override
	public void terminalError(Terminal.ErrorEvent event) {
		final Throwable t = event.getThrowable();
        if (t instanceof SocketException) {
            // Most likely client browser closed socket
            return;
        }

        // Finds the original source of the error/exception
        Object owner = null;
        if (event instanceof VariableOwner.ErrorEvent) {
            owner = ((VariableOwner.ErrorEvent) event).getVariableOwner();
        } else if (event instanceof URIHandler.ErrorEvent) {
            owner = ((URIHandler.ErrorEvent) event).getURIHandler();
        } else if (event instanceof ParameterHandler.ErrorEvent) {
            owner = ((ParameterHandler.ErrorEvent) event).getParameterHandler();
        } else if (event instanceof ChangeVariablesErrorEvent) {
            owner = ((ChangeVariablesErrorEvent) event).getComponent();
        }

        // Shows the error in AbstractComponent
        if (owner instanceof AbstractComponent && !(owner instanceof Button) ) { // We don't want to display error on Button. Only on components
            if (t instanceof ErrorMessage) {
                ((AbstractComponent) owner).setComponentError((ErrorMessage) t);
            } else {
                ((AbstractComponent) owner)
                        .setComponentError(new SystemError(t));
            }
        }

        // also print the error on console
        // log.error("Terminal error:", t);
		
        // ITCB hook for custom exception handler
		try  {
			currentPageMappingProcessor.handleError(event);
		}
		catch(Exception errorInExceptionManagementException) {
			errorInExceptionManagementException.printStackTrace();
		}
	}

	/**
	 * @return the currentPageMappingProcessor used to display page for this application context
	 */
	public PageMappingProcessor getCurrentPageMappingProcessor() {
		return currentPageMappingProcessor;
	}
	
	/**
	 * @return the window name. Has to be override in subclass if needed.
	 */
	protected String getWindowName() {
		return "";
	}
	
	/**
	 * @return an intermediate session : 
	 * <ul>
	 *   <li>not the servlet session which can be used in many applications</li>
	 *   <li>not the portlet session : same reason</li>
	 * </ul>
	 * This session is created at application init and is dying when application is released
	 */
	public ConcurrentMap<String, Object> getApplicationSession() {
		return applicationSession;
	}
	
	/**
	 * @return Spring application configuration which had been injected previously
	 */
	public ItcbVaadinApplicationConfig getApplicationConfig() {
		return itcbVaadinApplicationConfig;
	}
	
	
	
	/**
	 * @see ItcbVaadinPortletApplication#onRequestStart(HttpServletRequest, HttpServletResponse) : same mecanism for portlet
	 * @param request
	 * @param response
	 */
	@Override
	public void onRequestStart(PortletRequest request, PortletResponse response) {
		if(! (request instanceof ResourceRequest)  || ! ((ResourceRequest)request).getResourceID().equals("UIDL")) {
			String itcbmode = request.getPortletMode().toString();
			if(!isRunning() || (itcbmode != null && !itcbmode.equals(currentMode) && request instanceof RenderRequest )  ) {			
				initItcb(null, request, itcbmode);
				
				if(itcbVaadinApplicationConfig.getItcbInitRequestInterceptors() != null) {
					List<ItcbRequestInterceptor> list = itcbVaadinApplicationConfig.getItcbInitRequestInterceptors();
					for (int i = list.size(); i > 0; i--) {
						list.get(i-1).handleRequestEnd(this, request);
					}
				}
			}
			
			if(itcbVaadinApplicationConfig.getItcbRequestInterceptors() != null) {
				for (ItcbRequestInterceptor requestFilter : itcbVaadinApplicationConfig.getItcbRequestInterceptors()) {
					requestFilter.handleRequestStart(this, request);
				}
			}
			
			if(request instanceof RenderRequest) {
				currentPageMappingProcessor.refreshApplication();
			}
		}
	}

	/**
	 * @see ItcbVaadinPortletApplication#onRequestEnd(HttpServletRequest, HttpServletResponse) : same mecanism for portlet
	 * @param request
	 * @param response
	 */
	@Override
	public void onRequestEnd(PortletRequest request, PortletResponse response) {
		if(itcbVaadinApplicationConfig.getItcbRequestInterceptors() != null) {
			if(! (request instanceof ResourceRequest)  || ! ((ResourceRequest)request).getResourceID().equals("UIDL")) {
				List<ItcbRequestInterceptor> list = itcbVaadinApplicationConfig.getItcbRequestInterceptors();
				for (int i = list.size(); i > 0; i--) {
					list.get(i-1).handleRequestEnd(this, request);
				}
			}
		}
	}
	
	/**
	 * Each request in Servlet mode is received in this function.<br/>
	 * At first request (for the current application), init is done.<br/>
	 * {@link ItcbRequestInterceptor} configured in {@link ItcbVaadinApplicationConfig} are called 
	 * <ul>
	 *   <li>before and after init requests</li>
	 *   <li>before and after each request</li>
	 * </ul>
	 * depending on the list in which {@link ItcbRequestInterceptor} are injected in {@link ItcbVaadinApplicationConfig} bean.
	 * @param request
	 * @param response
	 */
	@Override
	public void onRequestStart(HttpServletRequest request,
			HttpServletResponse response) {	
		if(! request.getPathInfo().equals("/UIDL")) {
			String itcbmode = request.getParameter("itcbmode");
			if(itcbmode == null && currentMode == null) {
				itcbmode="view";
			}
			if(!isRunning() || ( itcbmode != null && !itcbmode.equals(currentMode) )   ) {			
				initItcb(request, null, itcbmode);
				if(itcbVaadinApplicationConfig.getItcbInitRequestInterceptors() != null) {
					List<ItcbRequestInterceptor> list = itcbVaadinApplicationConfig.getItcbInitRequestInterceptors();
					for (int i = list.size(); i > 0; i--) {
						list.get(i-1).handleRequestEnd(this, request);
					}
				}
			}
			
			if(itcbVaadinApplicationConfig.getItcbRequestInterceptors() != null) {
				for (ItcbRequestInterceptor requestFilter : itcbVaadinApplicationConfig.getItcbRequestInterceptors()) {
					requestFilter.handleRequestStart(this, request);
				}
			}
		}
		// TODO What about the renderrequest test
	}
	
	/**
	 * Called at the end of each request
	 * {@link ItcbRequestInterceptor#handleRequestEnd(ItcbVaadinServletApplication, HttpServletRequest)} methods are called if configured in {@link ItcbVaadinApplicationConfig}
	 * @param request
	 * @param response
	 */
	@Override
	public void onRequestEnd(HttpServletRequest request,
			HttpServletResponse response) {
		if(! request.getPathInfo().equals("/UIDL")) {
			if(itcbVaadinApplicationConfig.getItcbRequestInterceptors() != null) {
				List<ItcbRequestInterceptor> list = itcbVaadinApplicationConfig.getItcbRequestInterceptors();
				for (int i = list.size(); i > 0; i--) {
					list.get(i-1).handleRequestEnd(this, request);
				}
			}
		}
	}
	
}
