package net.sf.itcb.common.portlet.vaadin.page;

import java.util.Map;

import javax.portlet.PortletRequest;
import javax.servlet.ServletRequest;

import org.springframework.beans.factory.annotation.Required;

import net.sf.itcb.common.portlet.vaadin.AbstractItcbPortletApplication;
import net.sf.itcb.common.portlet.vaadin.component.ItcbComponent;
import net.sf.itcb.common.portlet.vaadin.component.ItcbPage;
import net.sf.itcb.common.portlet.vaadin.exception.ExceptionHandler;
import net.sf.itcb.common.portlet.vaadin.exception.ExceptionHandlerMapping;

import com.vaadin.terminal.Terminal.ErrorEvent;
import com.vaadin.ui.Panel;

/** 
 * PageMappingProcessor is the main controller for pages of the application.<br/>
 * The PageMappingProcessor's aim is to provide developers the toolkit they need to code vaadin pages and make them interact with each other.<br/>
 * Some properties have to set in Spring context : <code>mapping</code>, <code>defaultPageKey</code>, <code>exceptionHandlerMapping</code> (optional).<br/>
 * Some others are set by application : <code>componentContainer</code>, <code>request</code>, <code>application</code> <br/>
 * <code>currentPage</code> is used to manage the currently displayed page. It is also possible to display previous pages.<br/>
 * Developers can also use session storage (portal and portlet scope) and request management (portlet and servlet) 
 * without having to deel with portals heterogeneity.
 * 
 * @author Pierre Le Roux
 */
public interface PageMappingProcessor {

	public enum ReloadOrder {
		TRUE,
		FALSE
	}
	
	/**
	 * Define the pages mapping
	 * Has to be set by Spring context at creation
	 * @param mapping mapping that will be used for reach a page
	 */
	@Required
	public void setMapping(Map<String, ItcbComponent> mapping);
	

	/**
	 * Define the default page. This key has to be in mapping list
	 * Has to be set by Spring context at creation
	 * @param defaultPage the default page to show
	 */
	@Required
	public void setDefaultPageKey(String defaultPageKey) ;
	
	/**
	 * @param isAutomaticDisplay set the isAutomaticDisplay to true or false
	 */
	public void setIsAutomaticDisplay(Boolean isAutomaticDisplay);
	
	/**
	 * This mapping can be set by Spring in order to use specific mecanisms to display errors
	 * @param exceptionHandlerMapping
	 */
	public void setExceptionHandlerMapping(
			ExceptionHandlerMapping exceptionHandlerMapping);
	
	/**
	 * Defines if it is allowed to deal with fragment at application initialization.
	 * Applications which authorize it must deal with context in order to avoid not set variables...
	 * Application which unauthorize it, redirects the user on the same page or on the default page
	 * Default : false ==> Must be activated in Spring context
	 */
	public void setAuthorizeDirectAccessToFragmentOnLoad(boolean isAuthorizeFragmentOnLoad);
	
	// Application Specific params
	
	/**
	 * The request is set by the application at the initialization. It allows to deal with session and to get servlet request parameters.
	 * @param request
	 */
	public void setInitRequest(PortletRequest request);
	
	/**
	 * The request is set by the application at the initialization. It is used when user manage vaadin webapplication instead of portlet application
	 * @param request
	 */
	public void setInitRequest(ServletRequest request);
	
	/**
	 * @param componentContainer componentContainer to be set by the application : In most cases the componentContainer is the main window
	 * It's the componentContainer that has to be erased and refill by content
	 */
	public void setPanel(Panel panel);
	
	/**
	 * Set the current application object
	 * @param application
	 */
	public void setApplication(AbstractItcbPortletApplication application);

	/**
	 * Gets the application
	 * @return
	 */
	public AbstractItcbPortletApplication getApplication();
		
	/**
	 * Display the page associated with the id refPage. The page is simply displayed as it was the last time it has been displayed for this session
	 * If it is the first time, the page is initialized.
	 * If the <code>reload</code> parameter is set to <code>true</code>, page content is reloaded 
	 * @param refPage the page id referenced in <code>mapping</code>
	 * @param reload defines if the page has to be reloaded or not or only if modified
	 */
	public void displayPage(String refPage, ReloadOrder reload);
			
	/**
	 * Display the previous page
	 */
	public void displayPreviousPage();
	
	/**
	 * Display the default page defined in Spring context for the mapping processor
	 */
	public void displayDefaultPage() throws Exception;
	
	/**
	 * Store an attribute in session
	 * @param key
	 * @param value
	 */
	public void setSessionAttribute (String key, Object value);
	
	/**
	 * Get an object from his key in session 
	 * @param key
	 * @param requiredType
	 * @return 
	 */
	public <T> T getSessionAttribute(String key, Class<T> requiredType);
	
	/**
	 * Get an object from his key in session 
	 * @param key
	 * @param requiredType
	 * @return 
	 */
	public <T> T getSharedSessionAttribute(String key, Class<T> requiredType);
	
	/**
	 * Set a shared session attribute (the key must be auhorized to shared in portal session in portal properties).
	 * For example, in Liferay, it can be configured in portal-ext.properties by addind the key in session.shared.attributes
	 * @param key
	 * @param value
	 */
	public void setSharedSessionAttribute(String key, Object value);
	
	/**
	 * Return the portlet request if needed for preferences, attributes...
	 */
	public PortletRequest getRequest();
	
	/**
	 * Get the parameter value in PortletRequest or in ServletRequest if not found in PortletRequest
	 * @param key
	 * @return value
	 */
	public String getRequestParameter(String key);

	/**
	 * Defines if the pageMappingProcessor is used when the context (view, edit, help) is launched or if it is used for displaying pages with a manual launch
	 * by calling displayDefaultPage
	 * @return
	 */
	public boolean isAutomaticDisplay();
	
	/**
	 * Handle an exception that occured in pages.
	 * @see ExceptionHandler
	 * @param e the exception
	 */
	public void handleException(Throwable e);
	
	/**
	 * Handle an error according to the mapping defined in Spring
	 * @param error
	 */
	public void handleError(ErrorEvent error);
	
	/**
	 * Get the owner of an ErrorEvent depending on its type
	 * @param error
	 */
	public Object getErrorOwner(ErrorEvent event);
	
	/**
	 * Retrieve a page already loaded from the page id in the PageMappingProcessor.
	 * @param <T> type of page class
	 * @param refPage the id of the page in the PageMappingProcessor
	 * @param requiredType the type of page class
	 * @return the page with the specified type or null if the page has not been loaded yet
	 */
	public <T extends ItcbPage> T getPage(String refPage, Class<T> requiredType);
	
	/**
	 * This function allow to preload a page without displaying it for now.<br/>
	 * It can be usefull when you want to have good display time and 
	 * if your next pages content is not linked to current page content and user actions.<br/>
	 * Parameters are set as in {@link PageMappingProcessor#displayPage(String, ReloadOrder) 
	 * @param refPage 
	 * @param reload
	 */
	public void preloadPage(String refPage, ReloadOrder reload);


	public void setRefreshApplication();
	
	
}
