package net.sf.itcb.common.portlet.vaadin.exception.impl;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import com.vaadin.terminal.Terminal.ErrorEvent;

import net.sf.itcb.common.portlet.vaadin.exception.ExceptionHandler;
import net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor;
import net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor.ReloadOrder;

/**
 * This class handles exception in order to redirect to a generic error page 
 * that has to be defined by user and set in pageMappingProcessor settings<br/>
 * Developer must define the <code>errorPageKey</code> parameter 
 * and also can define the key in session in which the exception will be stored (using <code>exceptionInSessionKey</code>
 * @author Pierre Le Roux
 */
public class PageRedirectExceptionHandler implements ExceptionHandler {
	
	private String errorPageKey;
	private String exceptionInSessionKey;
	
	protected final Logger log = LoggerFactory.getLogger(getClass());
	
	@Required
	/**
	 * Set the error page to display
	 * @param errorPageKey the error page key
	 */
	public void setErrorPageKey(String errorPageKey) {
		this.errorPageKey = errorPageKey;
	}
	
	/**
	 * Set the session key in which the exception will be stored. If no session key is defined the exception won't be print and won't be stored in session
	 * @param exceptionInSessionKey the session key
	 */
	public void setExceptionInSessionKey (String exceptionInSessionKey){
		this.exceptionInSessionKey = exceptionInSessionKey;
	}
	
	
	
	/**
	 * Redirect to a generic error page after setting
	 * @param e the exception
	 * @param pageMappingProcessor the pageMappingProcessor
	 */
	public void handleError(ErrorEvent error, PageMappingProcessor pageMappingProcessor) {
		if(errorPageKey == null) {
			log.error("the errorPageKey must be declared in Spring configuration for this bean");
		}
		if(exceptionInSessionKey != null) {
			pageMappingProcessor.setSessionAttribute(exceptionInSessionKey, error.getThrowable().getCause());
			pageMappingProcessor.displayPage(errorPageKey, ReloadOrder.TRUE);
		}
		else {
			pageMappingProcessor.displayPage(errorPageKey, ReloadOrder.FALSE);
		}
	}

}
