package net.sf.itcb.common.portlet.vaadin;

import java.net.SocketException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.portlet.PortletMode;
import javax.portlet.PortletRequest;
import javax.portlet.PortletResponse;
import javax.portlet.RenderRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.sf.itcb.common.business.core.ItcbApplicationContextHolder;
import net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.Application;
import com.vaadin.terminal.ErrorMessage;
import com.vaadin.terminal.ParameterHandler;
import com.vaadin.terminal.SystemError;
import com.vaadin.terminal.Terminal;
import com.vaadin.terminal.URIHandler;
import com.vaadin.terminal.VariableOwner;
import com.vaadin.terminal.gwt.server.ChangeVariablesErrorEvent;
import com.vaadin.terminal.gwt.server.HttpServletRequestListener;
import com.vaadin.terminal.gwt.server.PortletRequestListener;
import com.vaadin.terminal.gwt.server.WebApplicationContext;
import com.vaadin.ui.AbstractComponent;
import com.vaadin.ui.Button;
import com.vaadin.ui.Panel;
import com.vaadin.ui.TabSheet;
import com.vaadin.ui.TabSheet.SelectedTabChangeEvent;
import com.vaadin.ui.TabSheet.SelectedTabChangeListener;
import com.vaadin.ui.TabSheet.Tab;
import com.vaadin.ui.VerticalLayout;
import com.vaadin.ui.Window;



/**
 * this abstract class is the base vaadin application portlet used by all the vaadin application
 * It manages common treatments :
 * <ul>
 * <li>user storage in a spring Context {@link org.springframework.security.core.context.SecurityContextHolder}</li>
 * <li>local storage in a spring Context {@link org.springframework.context.i18n.LocaleContextHolder}</li> 
 * <li>user storage in MDC context in order to be logged {@link org.slf4j.MDC}</li>
 * <li>create the application and all the views (VIEW, EDIT, HELP) when the application is run inside a portal or directly in an application server as a servlet</li>
 * <li>delegates the views contents to child classes</li>
 * </ul>
 * 
 * TODO (Feature request  3155816): Request in an portal free mode (running in Tomcat as servlet) is ok but 2 things has to be done :
 * <ul>
 * <li>Request are now RenderRequest but it has to be also possible to use HttpServletRequest</li>
 * <li>Users have to be mocked when we are in Servlet mode</li>
 * <li>Edit View and Help mode have to be displayed with a tabs screen instead of the 3 actual buttons.</li>
 * </ul>
 * In child class and all the other classes : Services, DAO... The user will be available by calling the spring context : SecurityContextHolder.getContext().getAuthentication().
 * 
 * @author Pierre Le Roux
 */
public abstract class AbstractItcbPortletApplication extends Application implements HttpServletRequestListener, PortletRequestListener  {
	
	private static final long serialVersionUID = 1L;
	
	protected final Logger log = LoggerFactory.getLogger(getClass());
	
	protected Window mainWindow;
	
	protected String preferencesURL;
	
	protected PageMappingProcessor viewModePageMappingProcessor;
	
	protected PageMappingProcessor editModePageMappingProcessor;
	
	protected PageMappingProcessor helpModePageMappingProcessor;
	
	protected PageMappingProcessor currentPageMappingProcessor;
	
	protected PortletMode currentPortletMode;
	
	private HttpServletRequest servletRequest = null;
	
	// Application session. It is needed because portlet sessions are kept even if application is reloaded. It means that old portlet session is the same for new application in the same session.
	// It shouldn't.
	// With this session, vars are stored for the application life time. If application is reloaded, applicationSession is renewed
	private ConcurrentMap<String, Object> applicationSession;
	
	@Override
	public void onRequestStart(HttpServletRequest request,
			HttpServletResponse response) {	
		this.servletRequest = request;
		
	}
	
	@Override
	public void onRequestEnd(HttpServletRequest request,
			HttpServletResponse response) {
		// Nothing
		
	}
	
	

	@Override
	public void onRequestStart(PortletRequest request, PortletResponse response) {
		if(!isRunning() || (!request.getPortletMode().equals(currentPortletMode) && request instanceof RenderRequest )  ) {			
			applicationSession = new ConcurrentHashMap<String, Object>();
			mainWindow = new Window(getWindowName());
			setMainWindow(mainWindow);
			
			if (PortletMode.VIEW.equals(request.getPortletMode())) {
				viewModePageMappingProcessor = ItcbApplicationContextHolder.getContext().getBean("viewModePageMappingProcessor", PageMappingProcessor.class);	
				viewModePageMappingProcessor.setInitRequest(request);
				currentPageMappingProcessor = viewModePageMappingProcessor;
			}
			else if (PortletMode.EDIT.equals(request.getPortletMode())) {
				editModePageMappingProcessor = ItcbApplicationContextHolder.getContext().getBean("editModePageMappingProcessor", PageMappingProcessor.class);
				currentPageMappingProcessor = editModePageMappingProcessor;
				editModePageMappingProcessor.setInitRequest(request);
			}
			else if (PortletMode.HELP.equals(request.getPortletMode())) {
				helpModePageMappingProcessor = ItcbApplicationContextHolder.getContext().getBean("helpModePageMappingProcessor", PageMappingProcessor.class);
				currentPageMappingProcessor = helpModePageMappingProcessor;
				helpModePageMappingProcessor.setInitRequest(request);
			}
			
			currentPortletMode=request.getPortletMode();
			currentPageMappingProcessor.setPanel(getComponentContainer(currentPageMappingProcessor.getRequest().getPortletMode()));
			currentPageMappingProcessor.setApplication(AbstractItcbPortletApplication.this);
			
			try {
				if(currentPageMappingProcessor.isAutomaticDisplay()) {
					currentPageMappingProcessor.displayDefaultPage();
				}
			} 
			catch (Exception e) {
				try  {
					currentPageMappingProcessor.handleException(e);
				}
				catch(Exception errorInExceptionManagementException) {
					errorInExceptionManagementException.printStackTrace();
				}
			}
			
			if(currentPageMappingProcessor == null) {
				log.error("the "+ request.getPortletMode() +" associated mapping variable has to be set in spring context. The bean name has to be : " + request.getPortletMode() + "ModePageMappingProcessor");
				return;
			}
		}
		
		if(request instanceof RenderRequest) {
			currentPageMappingProcessor.setRefreshApplication();
		}
		
	}

	@Override
	public void onRequestEnd(PortletRequest request, PortletResponse response) {
		// Nothing
	}
	

	
	@Override
	public void init() {
		// Nothing
	}

	@Override
	public void terminalError(Terminal.ErrorEvent event) {
		final Throwable t = event.getThrowable();
        if (t instanceof SocketException) {
            // Most likely client browser closed socket
            return;
        }

        // Finds the original source of the error/exception
        Object owner = null;
        if (event instanceof VariableOwner.ErrorEvent) {
            owner = ((VariableOwner.ErrorEvent) event).getVariableOwner();
        } else if (event instanceof URIHandler.ErrorEvent) {
            owner = ((URIHandler.ErrorEvent) event).getURIHandler();
        } else if (event instanceof ParameterHandler.ErrorEvent) {
            owner = ((ParameterHandler.ErrorEvent) event).getParameterHandler();
        } else if (event instanceof ChangeVariablesErrorEvent) {
            owner = ((ChangeVariablesErrorEvent) event).getComponent();
        }

        // Shows the error in AbstractComponent
        if (owner instanceof AbstractComponent && !(owner instanceof Button) ) { // We don't want to display error on Button. Only on components
            if (t instanceof ErrorMessage) {
                ((AbstractComponent) owner).setComponentError((ErrorMessage) t);
            } else {
                ((AbstractComponent) owner)
                        .setComponentError(new SystemError(t));
            }
        }

        // also print the error on console
        // log.error("Terminal error:", t);
		
		try  {
			currentPageMappingProcessor.handleError(event);
		}
		catch(Exception errorInExceptionManagementException) {
			errorInExceptionManagementException.printStackTrace();
		}
	}

	
	/**
	 * Manage a servletContext mode
	 * TODO Not totally available for now
	 */
	private void manageServletContext(WebApplicationContext ctx) {
		mainWindow.removeAllComponents();
		TabSheet modeTabs = new TabSheet();
		modeTabs.setSizeFull();
		
		final VerticalLayout viewPanel = new VerticalLayout();
		viewPanel.setSizeFull();
		final VerticalLayout editPanel = new VerticalLayout();
		editPanel.setSizeFull();
		final VerticalLayout helpPanel = new VerticalLayout();
		helpPanel.setSizeFull();
		modeTabs.addTab(viewPanel, PortletMode.VIEW.toString(), null);
		modeTabs.addTab(editPanel, PortletMode.EDIT.toString(), null);
		modeTabs.addTab(helpPanel, PortletMode.HELP.toString(), null);
		
		modeTabs.setSelectedTab(viewPanel);
		currentPageMappingProcessor = viewModePageMappingProcessor;
		modeTabs.addListener(new SelectedTabChangeListener() {

			private static final long serialVersionUID = 1L;

			@Override
			public void selectedTabChange(SelectedTabChangeEvent event) {
				TabSheet tabsheet = event.getTabSheet();
		        Tab tab = tabsheet.getTab(tabsheet.getSelectedTab());
			
				if (PortletMode.VIEW.equals(tab.getCaption())) {
					currentPageMappingProcessor = viewModePageMappingProcessor;
					//currentPageMappingProcessor.setPanel(viewPanel); // FIXME Change it to make it more modular with PanelSpecific stuffs
				}
				else if (PortletMode.EDIT.equals(tab.getCaption())) {
					currentPageMappingProcessor = editModePageMappingProcessor;
					//currentPageMappingProcessor.setPanel(editPanel); // FIXME Change it to make it more modular with PanelSpecific stuffs
				}
				else if (PortletMode.HELP.equals(tab.getCaption())) {
					currentPageMappingProcessor = helpModePageMappingProcessor;
					//currentPageMappingProcessor.setPanel(helpPanel); // FIXME Change it to make it more modular with PanelSpecific stuffs
				}
				
				currentPageMappingProcessor.setInitRequest(servletRequest);
				currentPageMappingProcessor.setApplication(AbstractItcbPortletApplication.this);
				try {
					if(currentPageMappingProcessor.isAutomaticDisplay()) {
						currentPageMappingProcessor.displayDefaultPage();
					}
				} 
				catch (Exception e) {
					try  {
						currentPageMappingProcessor.handleException(e);
					}
					catch(Exception errorInExceptionManagementException) {
						errorInExceptionManagementException.printStackTrace();
					}
				}
				
			}
			
		});
		
		
		mainWindow.addComponent(modeTabs);		
		
	}

	public PageMappingProcessor getCurrentPageMappingProcessor() {
		return currentPageMappingProcessor;
	}
	

	protected String getWindowName() {
		return "";
	}
	
	public ConcurrentMap<String, Object> getApplicationSession() {
		return applicationSession;
	}
	
	
	/**
	 * Retrieves the componentContainer in which the content will be displayed
	 * If the PortletApplication implements {@link PanelSpecificPortlet}, 
	 * it will display content in the specific panel depending on the portletMode
	 * @param portletMode the portlet mode
	 * @return the panel in which the content will be written
	 */
	protected Panel getComponentContainer(PortletMode portletMode) {
		if(this instanceof PanelSpecificPortlet) {
			if(PortletMode.VIEW.equals(portletMode)) {
				return ((PanelSpecificPortlet)this).getPanelView();
			}
			else if(PortletMode.EDIT.equals(portletMode)) {
				return ((PanelSpecificPortlet)this).getPanelEdit();
			}
			else if(PortletMode.HELP.equals(portletMode)) {
				return ((PanelSpecificPortlet)this).getPanelHelp();
			}
		}
		return getMainWindow();
	}
	
}
