package net.sf.itcb.common.portlet.vaadin.page.impl;

import java.io.Serializable;
import java.util.Map;

import javax.portlet.PortletRequest;
import javax.portlet.PortletSession;
import javax.servlet.ServletRequest;

import net.sf.itcb.common.business.core.ItcbApplicationContextHolder;
import net.sf.itcb.common.portlet.exceptions.PortletItcbException;
import net.sf.itcb.common.portlet.exceptions.PortletItcbExceptionMappingErrors;
import net.sf.itcb.common.portlet.portal.PortalAdapterProvider;
import net.sf.itcb.common.portlet.vaadin.AbstractItcbPortletApplication;
import net.sf.itcb.common.portlet.vaadin.exception.ExceptionHandlerMapping;
import net.sf.itcb.common.portlet.vaadin.page.AbstractItcbPortletController;
import net.sf.itcb.common.portlet.vaadin.page.AbstractItcbPortletPage;
import net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;

import com.vaadin.data.Validatable;
import com.vaadin.terminal.ParameterHandler;
import com.vaadin.terminal.Terminal.ErrorEvent;
import com.vaadin.terminal.URIHandler;
import com.vaadin.terminal.VariableOwner;
import com.vaadin.terminal.gwt.server.ChangeVariablesErrorEvent;
import com.vaadin.ui.Panel;
import com.vaadin.ui.ProgressIndicator;
import com.vaadin.ui.UriFragmentUtility;
import com.vaadin.ui.UriFragmentUtility.FragmentChangedEvent;
import com.vaadin.ui.UriFragmentUtility.FragmentChangedListener;

/**
 * ITCB implementation of PageMappingProcessor
 * 
 * @author Pierre Le Roux
 *
 */
public class PageMappingProcessorImpl implements PageMappingProcessor, Serializable {

	private static final long serialVersionUID = 1L;
	
	protected final Logger log = LoggerFactory.getLogger(getClass());
	
	Map<String, AbstractItcbPortletPage> mapping;
	
	protected String defaultPageKey;

	protected ExceptionHandlerMapping exceptionHandlerMapping;
	
	protected boolean authorizeDirectAccessToFragmentOnLoad=false;
	
	protected Panel panel;
	
	protected PortletRequest portletInitRequest;
	
	protected ServletRequest servletInitRequest;	
	
	protected AbstractItcbPortletApplication application;
	
	protected AbstractItcbPortletPage currentPage;
		
	protected static PortalAdapterProvider portalAdapterProvider = ItcbApplicationContextHolder.getContext().getBean("itcbPortalAdapterProvider", PortalAdapterProvider.class);
	
	protected PageMappingProcessorAsyncDelegate asyncDelegate = ItcbApplicationContextHolder.getContext().getBean("itcbPageMappingProcessorAsyncDelegate", PageMappingProcessorAsyncDelegate.class);
	
	protected UriFragmentUtility uriFragmentUtility;
	
	ProgressIndicator progressIndicator;
	
	protected String firstPageKey;
		
	/**
	 * By default, the default page is displayed at portlet initialization<br/>
	 * If the portlet doesn't want to use this processor at portlet initialization,
	 * Spring context has to be set to false in Spring configuration
	 */
	protected Boolean isAutomaticDisplay = true;

	private String currentPageRef;

	private boolean refreshApplication=true;
	
	
	public PageMappingProcessorImpl() {
		
	}
	
	
	// Getters / Setters
	
	@Override
	public void setMapping(Map<String, AbstractItcbPortletPage> mapping) {
		this.mapping = mapping;
	}
	
	@Override
	public void setDefaultPageKey(String defaultPageKey) {
		this.defaultPageKey = defaultPageKey;
	}
	
	@Override
	public void setIsAutomaticDisplay(Boolean isAutomaticDisplay) {
		this.isAutomaticDisplay = isAutomaticDisplay;
	}
	
	@Override
	public void setExceptionHandlerMapping(
			ExceptionHandlerMapping exceptionHandlerMapping) {
		this.exceptionHandlerMapping = exceptionHandlerMapping;
	}
	
	
	// Application Specific params
	
	@Override
	public void setInitRequest(PortletRequest portletRequest) {
		this.portletInitRequest = portletRequest;
		this.servletInitRequest = portalAdapterProvider.getPortalAdapter(portletRequest).getServletRequest(portletRequest);
	}
	
	@Override
	public void setInitRequest(ServletRequest servletRequest) {
		//MockPortletRequest mockRequest = new MockPortletRequest();
		// TODO Mock the portletRequest in order to make it available in servlet mode as if the user were in a portal
		//for (Iterator<Map.Entry<String,String>> iterator = servletRequest.getParameterMap().entrySet().iterator(); iterator.hasNext();) {
		//	Map.Entry<String, String> entry = (Map.Entry<String, String>) iterator.next();
			//mockRequest.addParameter(entry.getKey().toString(), entry.getValue().toString());
		//}
		//this.request = mockRequest;
		this.servletInitRequest=servletRequest;
	}
	
	@Override
	public void setPanel(Panel panel) {
		this.panel = panel;
	}
	
	@Override
	public void setApplication(AbstractItcbPortletApplication application) {
		this.application = application;
	}

	@Override
	public AbstractItcbPortletApplication getApplication() {
		return application;
	}
	
	// Functions
	
	@Override
	public void setAuthorizeDirectAccessToFragmentOnLoad(boolean authorizeDirectAccessToFragmentOnLoad) {
		this.authorizeDirectAccessToFragmentOnLoad= authorizeDirectAccessToFragmentOnLoad;	
	}
	
	@Override
	public void displayPage(String refPage, ReloadOrder reload) {
		log.debug("displayPage {} {}" , refPage, reload);
			if(reload != null) {
				changePage(refPage, reload);
				uriFragmentUtility.setFragment(refPage, false);
				  if(firstPageKey == null) {
						firstPageKey = refPage;
				  }
			}		
	}
	
	@Override
	public void displayDefaultPage() throws Exception {
		log.debug("Display the default page");
		if(defaultPageKey != null) {
			uriFragmentUtility = new UriFragmentUtility();
			uriFragmentUtility.addListener(new FragmentChangedListener() {
				   /**
				 * 
				 */
				private static final long serialVersionUID = 1L;

				public void fragmentChanged(FragmentChangedEvent source) {
				      if(currentPage instanceof Validatable) {
				    	  try  {
				    		  ((Validatable)currentPage).validate();
				    	  }
				    	  catch(Exception e) {
				    		  uriFragmentUtility.setFragment(currentPageRef, false);
				    		  handleException(e);
				    		  return;
				    	  }
				      }
					  String fragment = source.getUriFragmentUtility().getFragment();
				      if (fragment != null) {
					  	  log.debug("changeFragment {}", fragment);
						  String refPage="";
				    	  String[] fragmentParts = fragment.split("-");
					      refPage=fragmentParts[0];
						  if(refPage.equals("")) {
							  if(refreshApplication == true) {
								refPage = currentPageRef;
							  }
							  else {
								  refPage = firstPageKey;
							  }
							  log.debug("No fragment on call so calling {}", refPage);
						  }
						  if(!authorizeDirectAccessToFragmentOnLoad && !isAlreadyLoaded(refPage)) {
							log.debug("Direct access to {} not allowed", refPage);
							refPage = currentPageRef;
							uriFragmentUtility.setFragment(refPage, false);
							log.debug("Calling {} instead", refPage);
						  }
							
				    	changePage(refPage, ReloadOrder.FALSE);
				    	refreshApplication=false;
				      }
				   }

				private boolean isAlreadyLoaded(String refPage) {
					AbstractItcbPortletPage page = mapping.get(refPage);
					if(page != null && page.getParent() != null) {
						return true;
					}
					return false;
				}
				});
			panel.addComponent(uriFragmentUtility);
			AbstractItcbPortletPage page=  mapping.get(defaultPageKey);
			if(page instanceof AbstractItcbPortletController) {
				page.setPageMappingProcessor(this);
				page.removeAllComponents();
				page.initPage();
			}
			else {
				displayPage(defaultPageKey, ReloadOrder.TRUE);
			}
		}
		else {
			if(log.isDebugEnabled()) {
				log.warn("No default page configured. PageMappingProcessor mecanism not used. Set the defaultPageKey property in your spring context.");
			}
		}
	}
	
	@Override
	public void displayPreviousPage() {
		panel.getApplication().getMainWindow().executeJavaScript("history.back()");
	}
	
	@Override
	public void setSessionAttribute (String key, Object value) {
		if(value == null) {
			application.getApplicationSession().remove(key);
		}
		else {
			application.getApplicationSession().put(key, value);
		}
	}
	
	@Override
	@SuppressWarnings("unchecked")
	public <T> T getSessionAttribute(String key, Class<T> requiredType) {
		return (T)application.getApplicationSession().get(key );
	}
	
	@Override
	@SuppressWarnings("unchecked")
	public <T> T getSharedSessionAttribute(String key, Class<T> requiredType) {
		return (T) portletInitRequest.getPortletSession().getAttribute(key, PortletSession.APPLICATION_SCOPE);
	}
	
	@Override
	public void setSharedSessionAttribute(String key, Object value) {
		portletInitRequest.getPortletSession().setAttribute(key, PortletSession.APPLICATION_SCOPE);
	}
	
	@Override
	public PortletRequest getRequest() {
		return portletInitRequest;
	}

	@Override
	public boolean isAutomaticDisplay() {
		return isAutomaticDisplay;
	}
	
	@Override
	public void handleException(Throwable e) {
		if(e instanceof RuntimeException) {
			throw (RuntimeException)e;
		}
		else {
			MessageSource itcbPortletMessageSource = ItcbApplicationContextHolder.getContext().getBean("common-portletResources", MessageSource.class);
			throw new PortletItcbException(PortletItcbExceptionMappingErrors.COMMON_PORTLET_LOADING_PAGE, itcbPortletMessageSource.getMessage("common.portlet.exception.loading.page", new Object[] {e.getMessage()}, LocaleContextHolder.getLocale()));
		}
	}
	
	@Override
	public void handleError(ErrorEvent error) {
		try {
			if(exceptionHandlerMapping != null) {
				exceptionHandlerMapping.handleError(error, this);
			}
			else {
				Throwable t = error.getThrowable().getCause();
				if(t != null) {
					log.error(t.getMessage(), t);
				}
			}
		}
		catch(Exception exceptionInExceptionManagement) {
			log.error("The exception has not been handled well {}. This Exception occured {}", error.getThrowable(), exceptionInExceptionManagement);
		}
	}

	@Override
	public String getRequestParameter(String key) {
		String value = portletInitRequest.getParameter(key);
		if(value == null) {
			value=servletInitRequest.getParameter(key);
		}
		return value;
	}

	@Override
	public Object getErrorOwner(ErrorEvent event) {
		Object owner = null;
        if (event instanceof VariableOwner.ErrorEvent) {
            owner = ((VariableOwner.ErrorEvent) event).getVariableOwner();
        } else if (event instanceof URIHandler.ErrorEvent) {
            owner = ((URIHandler.ErrorEvent) event).getURIHandler();
        } else if (event instanceof ParameterHandler.ErrorEvent) {
            owner = ((ParameterHandler.ErrorEvent) event).getParameterHandler();
        } else if (event instanceof ChangeVariablesErrorEvent) {
            owner = ((ChangeVariablesErrorEvent) event).getComponent();
        }

        return owner;
		
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T extends AbstractItcbPortletPage> T getPage(String refPage,
			Class<T> requiredType) {
		return (T) mapping.get(refPage);
	}

	@Override
	/**
	 * This function is called with async method in order to construct the page serverside without time issue for the user
	 */
	public void preloadPage(String refPage, ReloadOrder reload) {
		if(progressIndicator==null) {
			progressIndicator = new ProgressIndicator();
			progressIndicator.setIndeterminate(true);
			progressIndicator.setPollingInterval(5000);
			//progressIndicator.setVisible(false);
			progressIndicator.addStyleName(PageMappingProcessorAsyncDelegate.HIDDEN_STYLE);
	        panel.addComponent(progressIndicator);
		}
		asyncDelegate.preloadPage(refPage, reload, this);
		progressIndicator.setEnabled(true);
		
	}
	
	/**
	 * This method is called in order to change page and to display it
	 * @param refPage
	 * @param reloadOrder
	 */
	protected void changePage(String refPage, ReloadOrder reloadOrder) {
		log.debug("changePage {} {}" , refPage, reloadOrder);
		try {
			AbstractItcbPortletPage page = mapping.get(refPage);
			if(page == null) {
				MessageSource itcbPortletMessageSource = ItcbApplicationContextHolder.getContext().getBean("common-portletResources", MessageSource.class);
				throw new PortletItcbException(PortletItcbExceptionMappingErrors.COMMON_PORTLET_MISSING_PAGE, itcbPortletMessageSource.getMessage("common.portlet.exception.missing.page", new Object[] {refPage}, LocaleContextHolder.getLocale()));
			}
			
			if( !(page instanceof AbstractItcbPortletController)) {
				if(currentPage != null && currentPage != page) {
					currentPage.setVisible(false);
				}
				
				if (page.getParent() == null || reloadOrder.equals(ReloadOrder.TRUE) ) {
					loadPage(page, reloadOrder);
				}
				// If the page has already been loaded and the page has a method to call to be updated each time and reloadOrder was FALSE
				// then we update the page according to Updatable.update method
				else if (page instanceof AbstractItcbPortletPage.Updatable) {
					((AbstractItcbPortletPage.Updatable)page).update();
				}
				page.setVisible(true);
				
				// If the page has already been preloaded, we remove the hidden style
				page.removeStyleName(PageMappingProcessorAsyncDelegate.HIDDEN_STYLE);
				currentPage=page;
				currentPageRef=refPage;
			}
		
		}
		catch(RuntimeException re) {
			throw re;
		}
		catch(final Exception e) {
			MessageSource itcbPortletMessageSource = ItcbApplicationContextHolder.getContext().getBean("common-portletResources", MessageSource.class);
			throw new PortletItcbException(PortletItcbExceptionMappingErrors.COMMON_PORTLET_LOADING_PAGE, itcbPortletMessageSource.getMessage("common.portlet.exception.loading.page", new Object[] {refPage, e.getMessage()}, LocaleContextHolder.getLocale()));
		}
		
	}
	
	/**
	 * Load the page or reload it and attach it to the main panel (window)
	 * This function can be called either by {@link PageMappingProcessor#displayPage(String, net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor.ReloadOrder)} or {@link PageMappingProcessor#preloadPage(String, net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor.ReloadOrder)}
	 * @param page
	 * @param reload
	 * @throws Exception
	 */
	protected void loadPage(AbstractItcbPortletPage page, ReloadOrder reload)
			throws Exception {
		log.debug("loadPage {} {}" + page, reload);
		if(page != null && 
				(reload == ReloadOrder.TRUE)) {
			page.setPageMappingProcessor(this);
			page.removeAllComponents();
			if(page.getParent() == null) {
				panel.addComponent(page);
			}
			page.initPage();
			
		}
		if(page.getPageMappingProcessor() == null || page.getParent() == null || (!page.getComponentIterator().hasNext() && !(page instanceof AbstractItcbPortletController) ) ) {
			page.setPageMappingProcessor(this);
			page.initPage();
			panel.addComponent(page);
		}
	}


	@Override
	public void setRefreshApplication() {
		this.refreshApplication= true;
		
	}


	
}
