package net.sf.itcb.common.portlet.vaadin.page;


import java.util.HashMap;
import java.util.Map;

import net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor.ReloadOrder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.data.Property.ValueChangeEvent;
import com.vaadin.data.Property.ValueChangeListener;
import com.vaadin.data.Property.ValueChangeNotifier;
import com.vaadin.ui.VerticalLayout;

/**
 * Each page has to extend this class
 * 
 * @author Pierre Le Roux
 *
 */
public abstract class AbstractItcbPortletPage extends VerticalLayout {

	private static final long serialVersionUID = 1L;

	private PageMappingProcessor pageMappingProcessor;
	
	private Map<ValueChangeNotifier, Boolean> listenedFieldsForChanges;
	
	private FieldValueChangeListener fieldValueChangeListener;
	
	protected final Logger log = LoggerFactory.getLogger(getClass());
		
	public PageMappingProcessor getPageMappingProcessor() {
		return pageMappingProcessor;
	}
	
	public void initPage() throws Exception {
		log.debug("Initializing {}", getClass());
		removeAllComponents();
		setMargin(false);
		setSpacing(false);
		defineContentDelegate();
		if(this instanceof FieldsValuePreLoadable) {
			((FieldsValuePreLoadable)this).fillFieldsWithValues();
		}
	}
	
	protected abstract void defineContentDelegate() throws Exception;

	public void setPageMappingProcessor(
			PageMappingProcessor pageMappingProcessor) {
		this.pageMappingProcessor = pageMappingProcessor;
	}
	
	/**
	 * This method allows to add a field in the list which will be parsed when the user will click in order to change page ({@link PageMappingProcessor#displayPage(String, net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor.ReloadOrder)})<br/>
	 * Actually, this method is called only if {@link PageMappingProcessor#displayPage(String, net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor.ReloadOrder)} is called with attribute {@link ReloadOrder#IF_MODIFIED)}
	 * @param valueChangeNotifier
	 */
	public void addFieldToListenForChange(ValueChangeNotifier component) {
		if(listenedFieldsForChanges==null) {
			listenedFieldsForChanges=new HashMap<ValueChangeNotifier, Boolean>();
			fieldValueChangeListener = new FieldValueChangeListener();
		}
		listenedFieldsForChanges.put(component, Boolean.FALSE);
		component.addListener(fieldValueChangeListener);
	}
	
	/**
	 * return <code>true</code> if one of {@link AbstractItcbPortletPage#listenedFieldsForChanges} changed since the last time the page has been displayed
	 * @return true or false
	 */
	public boolean hasChangedSinceLastDisplay() {
		if(listenedFieldsForChanges != null) {
			for (Boolean fieldHasChanged : listenedFieldsForChanges.values()) {
				if(fieldHasChanged) {
					return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * This interface has to be implemented by any page which can be preloaded with default values
	 * Developers have to code default values logic in an overrided {@link FieldsValuePreLoadable#fillFieldsWithValues()} method
	 * @author Pierre Le Roux
	 */
	public interface FieldsValuePreLoadable {
		/**
		 * Method that has to be overrided in order to fill fields when the page is loaded
		 */
		public void fillFieldsWithValues();
	}
	
	
	/**
	 * The fieldValueChangeListener is adding by {@link AbstractItcbPortletPage} for fields which changes could occurate a reload of next pages
	 * @author Pierre Le Roux
	 */
	protected class FieldValueChangeListener implements ValueChangeListener{
		private static final long serialVersionUID = 1L;

		@Override
		public void valueChange(ValueChangeEvent event) {
			listenedFieldsForChanges.put((ValueChangeNotifier)event.getProperty(), Boolean.TRUE);
		}
	}
	
	
	/**
	 * This interface has to be implemented by any page which has to be updated on each displayPage if delegateContent is not called once again<br/>
	 * We can meet this situation if {@link PageMappingProcessor#displayPage(String, net.sf.itcb.common.portlet.vaadin.page.PageMappingProcessor.ReloadOrder)} 
	 * function is called with {@link ReloadOrder#IF_MODIFIED} or {@link ReloadOrder#FALSE} <br/>
	 * It is usefull to keep fields values as typed by user but update a part of screen using model or session updated information.<br/>
	 * The update function has to be implemented by subclasses in order to update a part of screen.
	 * @author Pierre Le Roux
	 *
	 */
	public interface Updatable {
		public void update();
	}
	   	
}

