package net.sf.filePiper.processors;


import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.Arrays;

import net.sf.sfac.gui.editor.ObjectEditor;
import net.sf.filePiper.gui.SizeAndUnitEditor;
import net.sf.filePiper.model.FileProcessorEnvironment;
import net.sf.filePiper.model.OneToOneByteFileProcessor;


/**
 * Processor copying just the x last lines (or bytes) of a file.
 * 
 * @author BEROL
 */
public class TailProcessor extends OneToOneByteFileProcessor implements SizeAndUnit {


    private static final String TAIL_SIZE = "tail.size";
    private static final String TAIL_UNITS = "tail.units";


    public String getProcessorName() {
        return "Tail";
    }


    public int getSize() {
        return getSettings().getIntProperty(TAIL_SIZE, 100);
    }


    public void setSize(int newSize) {
        getSettings().setIntProperty(TAIL_SIZE, newSize);
    }


    public int getUnits() {
        return getSettings().getIntProperty(TAIL_UNITS, UNIT_LINE);
    }


    public void setUnits(int newUnits) {
        getSettings().setIntProperty(TAIL_UNITS, newUnits);
    }


    public String getProposedNameSuffix() {
        return "tail";
    }


    @Override
    public void process(InputStream is, OutputStream os, FileProcessorEnvironment env) throws IOException {
        if (getUnits() == UNIT_BYTE) {
            processBytes(is, os, env);
        } else {
            BufferedReader br = new BufferedReader(new InputStreamReader(is));
            BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(os));
            processLines(br, bw, env);
            bw.close();
        }
    }


    public void processBytes(InputStream is, OutputStream os, FileProcessorEnvironment env) throws IOException {
        int size = getSize();
        int[] bytes = new int[size];
        Arrays.fill(bytes, -1);
        int readByte;
        int i = 0;
        // put the bytes in a rolling buffer
        while (((readByte = is.read()) >= 0) && env.shouldContinue()) {
            bytes[i] = readByte;
            bytesProcessed(1);
            i++;
            if (i >= size) i = 0;
        }
        // write the rolling buffer to output stream
        for (int j = i; (j < size) && env.shouldContinue(); j++) {
            if (bytes[j] >= 0) {
                os.write(bytes[j]);
            }
        }
        for (int j = 0; (j < i) && env.shouldContinue(); j++) {
            if (bytes[j] >= 0) {
                os.write(bytes[j]);
            }
        }
    }


    public void processLines(BufferedReader in, BufferedWriter out, FileProcessorEnvironment env) throws IOException {
        int size = getSize();
        String[] lines = new String[size];
        String line;
        int i = 0;
        // put the lines in a rolling buffer
        while (((line = in.readLine()) != null) && env.shouldContinue()) {
            lines[i] = line;
            linesProcessed(1);
            i++;
            if (i >= size) i = 0;
        }
        // write the rolling buffer to output stream
        for (int j = i; (j < size) && env.shouldContinue(); j++) {
            if (lines[j] != null) {
                out.write(lines[j]);
                out.newLine();
            }
        }
        for (int j = 0; (j < i) && env.shouldContinue(); j++) {
            if (lines[j] != null) {
                out.write(lines[j]);
                out.newLine();
            }
        }
    }


    public ObjectEditor getEditor() {
        return new SizeAndUnitEditor("Output the X last lines/bytes of the input file", "Tail size");
    }

}
