package net.sf.filePiper.processors;


import java.io.BufferedReader;
import java.io.BufferedWriter;

import net.sf.filePiper.model.FileProcessorEnvironment;
import net.sf.filePiper.model.OneToOneTextFileProcessor;
import net.sf.sfac.editor.EditorConfig;
import net.sf.sfac.setting.Settings;
import net.sf.sfac.utils.Comparison;


/**
 * Processor replacing a pattern by line breaks.
 * 
 * @author BEROL
 */
public class BreakLinesProcessor extends OneToOneTextFileProcessor {


    private static final String BREAK_PATTERN = "break.pattern";
    private static final String KEEP_BREAK_PATTERN = "break.keep.pattern";

    private Settings sett;


    @Override
    public String getProcessorName() {
        return "Break lines";
    }


    @Override
    public String getProcessorDescription() {
        return "Break lines when encountering a defined pattern.";
    }


    @EditorConfig(label = "Break at", index = 0)
    public String getBreakString() {
        return sett.getStringProperty(BREAK_PATTERN, "");
    }


    public void setBreakString(String searchString) {
        sett.setStringProperty(BREAK_PATTERN, searchString);
    }


    @EditorConfig(label = "Keep break string", index = 1)
    public boolean getKeepBreakString() {
        return sett.getBooleanProperty(KEEP_BREAK_PATTERN, true);
    }


    public void setKeepBreakString(boolean newKeepBreakString) {
        sett.setBooleanProperty(KEEP_BREAK_PATTERN, newKeepBreakString);
    }


    @Override
    public void init(Settings settgns) {
        sett = settgns;
    }


    @Override
    public void process(BufferedReader in, BufferedWriter out, FileProcessorEnvironment env) throws Exception {
        StringBuilder sb = new StringBuilder();
        String breakString = getBreakString();
        boolean keep = getKeepBreakString();
        if (Comparison.isEmpty(breakString)) throw new IllegalStateException("You must define a break string");
        int ch;
        while ((ch = in.read()) >= 0) {
            sb.append((char) ch);
            if (endsWidth(sb, breakString)) {
                int len = sb.length();
                if (!keep) sb.setLength(len - breakString.length());
                out.write(sb.toString());
                out.newLine();
                sb.setLength(0);
            }
        }
    }


    private boolean endsWidth(StringBuilder sb, String str) {
        int strLen = str.length();
        int bufferOffset = sb.length() - strLen;
        if (bufferOffset < 0) return false;
        for (int i = 0; i < strLen; i++) {
            char ch = str.charAt(i);
            if (ch != sb.charAt(bufferOffset + i)) {
                return false;
            }
        }
        return true;
    }


    @Override
    public String getProposedNameSuffix() {
        return "break";
    }

}
