/*-------------------------------------------------------------------------
 Copyright 2006 Olivier Berlanger

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 -------------------------------------------------------------------------*/
package net.sf.filePiper;


import java.awt.BorderLayout;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.net.MalformedURLException;

import net.sf.filePiper.gui.PiperMainPanel;
import net.sf.filePiper.model.ToolModel;
import net.sf.sfac.gui.ExceptionDialog;
import net.sf.sfac.gui.framework.ActionRepository;
import net.sf.sfac.gui.framework.ApplicationFrame;
import net.sf.sfac.gui.framework.LookAndFeelHelper;
import net.sf.sfac.gui.framework.SharedResources;
import net.sf.sfac.gui.profiles.ProfilesController;
import net.sf.sfac.setting.Settings;
import net.sf.sfac.setting.SettingsImpl;

import org.apache.log4j.Logger;
import org.apache.log4j.xml.DOMConfigurator;


/**
 * Main class of the 'Piper file processing' tool. This class is a Builder building the editor tool in a frame to run as an
 * application.
 * <p>
 * This class interpret a simple command line:
 * 
 * <pre>
 *              java (jvm args) net.sf.filePiper.PiperTool [configFile]
 * </pre>
 * 
 * where:
 * <ul>
 * <li>configFile = path to the tool configuration file (default = './PiperTool.properties')
 * </ul>
 */
public class PiperTool {


    private static Logger log = Logger.getLogger(PiperTool.class);

    private ToolModel mainModel;

    private ApplicationFrame frame;

    private PiperMainPanel mainPanel;


    /**
     * Configure log4j using the "log4j.xml" file from the default (bin) directory.
     */
    public static void setupLogging() {
        try {
            DOMConfigurator.configure(new File("log4j.xml").toURI().toURL());
        } catch (MalformedURLException mue) {
            throw new IllegalStateException("Invalid log4j configuration file URL", mue);
        }
    }


    public PiperTool() {
    }


    private void buildMainFrame(File configFile) {
        try {
            // Build the Model part
            log.debug("Building model ...");
            Settings sett = new SettingsImpl(configFile, "Configuration file for the PiperTool");
            sett.load();
            mainModel = new ToolModel(sett);
            // build generic UI context
            log.debug("Building GUI ...");
            ActionRepository repo = new ActionRepository();
            LookAndFeelHelper lookHelper = new LookAndFeelHelper(sett);
            lookHelper.createActions(repo, "View/Look & Feel", "d/x9", null);
            // Build the specific UI part
            frame = new ApplicationFrame("File Piper", sett);
            frame.setIconImage(SharedResources.getIcon("frameIcon.gif").getImage());
            mainPanel = new PiperMainPanel(mainModel, repo);
            new ProfilesController(mainPanel, repo);
            // setup and show the frame
            log.debug("Show main frame ...");
            lookHelper.setRootUiComponent(frame);
            frame.getContentPane().add(mainPanel, BorderLayout.CENTER);
            frame.addWindowListener(new WindowAdapter() {


                @Override
                public void windowClosing(WindowEvent e) {
                    exit();
                }
            });
            frame.createViewToolbarAction(repo, "View", "d/a1", null);
            frame.setJMenuBar(repo.createMenuBar());
            frame.setToolBar(repo.createToolBar());
            frame.setVisible(true);
        } catch (Exception e) {
            ExceptionDialog.showExceptionDialog(frame, "Fatal Error", "Unable to start application", e);
        }
    }


    protected void exit() {
        mainPanel.exit();
    }


    static public void main(String[] args) {
        // configure static stuff
        setupLogging();
        SharedResources.addBaseImageLocation("/net/sf/filePiper/images");
        // MultiBorderLayout.debugMode = true;

        // get the configuration file
        String configFileName;
        int nbrArgs = (args == null) ? 0 : args.length;

        // get configuration file name
        if (nbrArgs > 0) configFileName = args[0];
        else configFileName = "PiperTool.properties";
        File configFile = new File(configFileName);
        log.info("Starting Piper using config file: " + configFile.getAbsolutePath());

        // start to build the DD editor tool
        PiperTool theTool = new PiperTool();
        theTool.buildMainFrame(configFile);
    }

}
