package net.sf.filePiper.processors;


import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import net.sf.filePiper.model.ExecutionPhase;
import net.sf.filePiper.model.FileProcessor;
import net.sf.filePiper.model.FileProcessorEnvironment;
import net.sf.filePiper.model.InputFileInfo;
import net.sf.filePiper.model.StatusHolder;
import net.sf.sfac.file.FilePathUtils;
import net.sf.sfac.gui.editor.ObjectEditor;
import net.sf.sfac.gui.editor.cmp.ReadOnlyObjectEditor;
import net.sf.sfac.setting.Settings;

import org.apache.log4j.Logger;


/**
 * Processor zipping all input streams in one output zip.
 * 
 * @author BEROL
 */
public class UnzipProcessor implements FileProcessor {


    Logger log = Logger.getLogger(UnzipProcessor.class);

    private StatusHolder holder = new StatusHolder() {


        @Override
        protected String getRunningMessage() {
            StringBuilder sb = new StringBuilder();
            sb.append("Unzipping ");
            appendCount(getInputFileCount(), "zip", sb);
            sb.append(" into ");
            appendCount(getOutputFileCount(), "file", sb);
            sb.append(" (");
            appendCount(getByteCount(), "byte", sb);
            sb.append(")...");
            return sb.toString();
        }


        @Override
        protected String getDoneMessage() {
            StringBuilder sb = new StringBuilder();
            appendCount(getInputFileCount(), "zip", sb);
            sb.append(" unzipped in ");
            appendCount(getOutputFileCount(), "file", sb);
            sb.append(" (");
            appendCount(getByteCount(), "byte", sb);
            sb.append(").");
            return sb.toString();
        }
    };


    public String getProcessorName() {
        return "Unzip";
    }


    public void init(Settings sett) {
    }


    public int getOutputCardinality(int inputCardinality) {
        return MANY;
    }


    public void process(InputStream is, InputFileInfo info, FileProcessorEnvironment env) throws IOException {
        ZipInputStream zipIs = new ZipInputStream(is);
        holder.inputFileStarted();
        ZipEntry entry = zipIs.getNextEntry();
        String zipInputPath = info.getInput().getAbsolutePath();
        // System.out.println("----- Input = " + zipInputPath);
        String zipDirectory = FilePathUtils.getDirectoryPath(zipInputPath);
        if (info.getBasePath() == null) info.setBasePath(new File(zipDirectory));
        while ((entry != null) && env.shouldContinue()) {
            if (!entry.isDirectory()) {
                String entryPath = entry.getName();
                File inputFile = new File(zipInputPath + "#" + entryPath);
                String entryParentPath = FilePathUtils.getDirectoryPath(entryPath);
                String name = FilePathUtils.getFileName(entryPath);
                String extension = FilePathUtils.getExtension(name);
                if (extension != null) name = name.substring(0, name.length() - (extension.length() + 1));
                // System.out.println("  * " + entryPath + " --> " + FilePathUtils.concatPaths(zipDirectory, entryParentPath)
                // + " + " + name + " + " + extension);
                info.setInput(inputFile);
                info.setProposedPath(FilePathUtils.concatPaths(zipDirectory, entryParentPath));
                info.setProposedName(name);
                info.setProposedExtension(extension);
                OutputStream os = env.getOutputStream(info);
                holder.outputFileStarted();
                byte[] buffer = new byte[1024];
                int len;
                while (((len = zipIs.read(buffer)) > 0) && env.shouldContinue()) {
                    os.write(buffer, 0, len);
                    holder.bytesProcessed(len);
                }
                os.close();
            }
            entry = zipIs.getNextEntry();
        }
    }


    public ObjectEditor getEditor() {
        return new ReadOnlyObjectEditor("Unzip all the input files.");
    }


    public void startBatch(FileProcessorEnvironment env) {
        holder.reset(ExecutionPhase.STARTING);
    }


    public void endBatch(FileProcessorEnvironment env) {
        holder.setCurrentPhase(env.getCurrentPhase());
    }


    public String getStatusMessage() {
        return holder.getStatusMessage();
    }


}
