package net.sf.filePiper.processors;


import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;

import net.sf.sfac.gui.editor.ObjectEditor;
import net.sf.sfac.gui.editor.cmp.ReadOnlyObjectEditor;
import net.sf.filePiper.model.ExecutionPhase;
import net.sf.filePiper.model.FileProcessor;
import net.sf.filePiper.model.FileProcessorEnvironment;
import net.sf.filePiper.model.InputFileInfo;
import net.sf.filePiper.model.Pipeline;
import net.sf.filePiper.model.StatusHolder;
import net.sf.sfac.setting.Settings;


/**
 * Processor listing the name of the input files.
 * 
 * @author BEROL
 */
public class ListFilesProcessor implements FileProcessor {


    private BufferedWriter out;
    private StatusHolder holder = new StatusHolder() {


        @Override
        protected String getRunningMessage() {
            StringBuilder sb = new StringBuilder();
            sb.append("Listing ");
            appendCount(getInputFileCount(), "file", sb);
            sb.append("...");
            return sb.toString();
        }


        @Override
        protected String getDoneMessage() {
            StringBuilder sb = new StringBuilder();
            appendCount(getInputFileCount(), "file", sb);
            sb.append(" listed.");
            return sb.toString();
        }
    };


    public String getProcessorName() {
        return "List files";
    }


    public void init(Settings settgns) {
    }


    public void startBatch(FileProcessorEnvironment env) throws IOException {
        holder.reset(ExecutionPhase.STARTING);
        Pipeline line = env.getPipeline();
        File dir = line.getSource();
        if (dir.isFile()) dir = dir.getAbsoluteFile().getParentFile();
        InputFileInfo info = new InputFileInfo(dir, new File(dir, "FileList.txt"));
        OutputStream os = env.getOutputStream(info);
        out = new BufferedWriter(new OutputStreamWriter(os));
        if (line.isSourceMultiFile()) {
            out.write("Files in ");
            out.write(dir.getAbsolutePath());
            out.newLine();
            String includes = line.getIncludesPattern();
            if ((includes != null) && !includes.trim().equals("")) {
                out.write("Including ");
                out.write(includes);
                out.newLine();
            }
            String excludes = line.getExcludesPattern();
            if ((excludes != null) && !excludes.trim().equals("")) {
                out.write("Excluding ");
                out.write(excludes);
                out.newLine();
            }
        } else {
            out.write("File ");
            out.write(line.getSource().getAbsolutePath());
            out.newLine();
        }
    }


    public void process(InputStream is, InputFileInfo info, FileProcessorEnvironment env) throws IOException {
        holder.inputFileStarted();
        out.write("    ");
        out.write(info.getInputRelativePath());
        out.newLine();
    }


    public void endBatch(FileProcessorEnvironment env) throws IOException {
        out.write("        = ");
        out.write(String.valueOf(holder.getInputFileCount()));
        out.write(" file");
        if (holder.getInputFileCount() > 1) out.write("s");
        out.write(".");
        out.newLine();
        out.close();
        out = null;
        holder.setCurrentPhase(env.getCurrentPhase());
    }


    public String getStatusMessage() {
        return holder.getStatusMessage();
    }


    public int getOutputCardinality(int inputCardinality) {
        return ONE;
    }


    public ObjectEditor getEditor() {
        return new ReadOnlyObjectEditor("List files of a directory");
    }

}
