package net.sf.filePiper.model;


import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import net.sf.sfac.setting.Settings;
import net.sf.sfac.setting.SubSettingsList;
import net.sf.sfac.setting.SubSettingsProxy;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import sun.misc.Service;


public class ToolModel {


    private static Log log = LogFactory.getLog(ToolModel.class);

    private Settings sett;
    private SubSettingsProxy pipelineSettings;
    private Pipeline pipeline;
    private SubSettingsList storedSettings;
    private List<FileProcessor> availableProcessors;
    private List<ChangeListener> listeners;


    public ToolModel(Settings setts) {
        sett = setts;
        storedSettings = new SubSettingsList(sett, "stored");
        pipelineSettings = new SubSettingsProxy(sett, "currentPipeline");
        log.info("Initialize pipeline from settings");
        pipeline = new Pipeline(pipelineSettings);
    }


    public Settings getSettings() {
        return sett;
    }


    private SubSettingsProxy getSubProfile(String profileName) {
        SubSettingsProxy subSett = null;
        int len = storedSettings.getSize();
        for (int i = 0; i < len; i++) {
            subSett = (SubSettingsProxy) storedSettings.getSubSettingAt(i);
            if (profileName.equals(subSett.getName())) {
                return subSett;
            }
        }
        return null;
    }


    public void storeCurrentProfile(String profileName) {
        SubSettingsProxy subSet = getSubProfile(profileName);
        if (subSet == null) {
            log.info("Store new profile with name:" + profileName);
            subSet = (SubSettingsProxy) storedSettings.addSubSetting();
            subSet.setName(profileName);
            storedSettings.setSelectedItem(subSet);
        } else {
            log.info("Update profile '" + profileName + "'");
            subSet.clear();
            subSet.setName(profileName);
        }
        subSet.copyValues(pipelineSettings);
    }


    public void changeProfileName(String oldName, String newName) {
        SubSettingsProxy subSet = getSubProfile(oldName);
        if (subSet != null) {
            subSet.setName(newName);
            storedSettings.sort();
        }
    }


    public void removeProfile(String profileName) {
        SubSettingsProxy subSett = getSubProfile(profileName);
        if (subSett != null) {
            storedSettings.removeSubSettings(subSett);
        }
    }


    public void useProfile(String profileName) {
        SubSettingsProxy subSett = getSubProfile(profileName);
        if (subSett != null) {
            pipelineSettings.clear();
            pipelineSettings.copyValues(subSett);
            pipelineSettings.removeProperty(SubSettingsProxy.SUB_SETTING_NAME);
            log.info("Copy pipeline settings from profile: " + profileName);
            pipeline.reset();
            fireDataChanged();
        }
    }


    public Pipeline getPipeline() {
        return pipeline;
    }


    @SuppressWarnings("unchecked")
    public List<FileProcessor> getAvailableProcessors() {
        if (availableProcessors == null) {
            availableProcessors = new ArrayList<FileProcessor>();
            log.info("SPI retrieving of all FileProcessor");
            Iterator<FileProcessor> plugInDefs = Service.providers(FileProcessor.class, getClass().getClassLoader());
            while (plugInDefs.hasNext()) {
                FileProcessor processor = plugInDefs.next();
                log.info("  * FileProcessor = " + processor.getClass());
                availableProcessors.add(processor);
            }
            Collections.sort(availableProcessors, new Comparator<FileProcessor>() {


                public int compare(FileProcessor fp1, FileProcessor fp2) {
                    return fp1.getProcessorName().compareTo(fp2.getProcessorName());
                }

            });
        }
        return availableProcessors;
    }


    public FileProcessor getPrototype(FileProcessor fp) {
        FileProcessor proto = null;
        for (FileProcessor processor : getAvailableProcessors()) {
            if (processor.getClass() == fp.getClass()) {
                proto = processor;
                break;
            }
        }
        return proto;
    }


    public SubSettingsList getStoredSettings() {
        return storedSettings;
    }


    public void addChangeListener(ChangeListener l) {
        if (listeners == null) listeners = new ArrayList<ChangeListener>();
        listeners.add(l);
    }


    public void removeChangeListener(ChangeListener l) {
        if (listeners != null) listeners.remove(l);
    }


    public void fireDataChanged() {
        if (listeners != null) {
            ChangeEvent ev = new ChangeEvent(this);
            for (ChangeListener l : listeners) {
                l.stateChanged(ev);
            }
        }
    }


}
