package net.sf.filePiper.model;


import java.io.File;
import java.io.OutputStream;


/**
 * Interface used by the whole pipeline to interact with its environment. <br>
 * It's used to:
 * <ul>
 * <li>Report current processing state (what has been processed yet).
 * <li>Get external abort signal.
 * <li>Check if a file can be overwritten.
 * </ul>
 * 
 * @author BEROL
 */
public interface PipelineEnvironment {


    /**
     * Notify the environment that the pipe-line processing has started.
     */
    public void startProcessing();


    /**
     * Notify the environment that the pipeline is scanning directory to find file to process.
     */
    public void scanningDirectory(File dir);


    /**
     * Notify the environment of the currently processed input file.
     */
    public void processInputFile(File src);


    /**
     * Notify the environment of the currently processed output file.
     */
    public void outputtingToFile(File dest);


    /**
     * Notify the environment of the currently processed output file is not written to avoid overriding an existing file.
     */
    public void fileSkipped(File dest);


    /**
     * Notify the environment that the pipe-line processing is finished.
     * 
     * @param e
     *            if not null, it means that the processing was stopped by this exception.
     */
    public void finished(Exception e);


    /**
     * Check if the processing should continue. The environment can decide to abort the pipe-line processing by returning false.
     * 
     * @return true to continue, false to abort.
     */
    public boolean shouldContinue();


    /**
     * Check if an existing file can be overwritten by the pipe-line processing.
     * 
     * @param fil
     *            the existing file that is about to be overwritten.
     * @return true to overwrite, false to skip.
     */
    public boolean canOverwriteFile(File fil);


    /**
     * Tell this environment that the current file processing should be aborted. <br>
     * Once this method is called, <code>shouldContinue()</code> must return false.
     */
    public void abortProcessing();


    /**
     * Get an output stream to write to the console.
     * 
     * @return an output stream to write to the console.
     */
    public OutputStream getConsoleStream();


    public boolean isRunning();


    public boolean isAborted();


    public boolean isErrored();


}
