package net.sf.filePiper.model;


import java.io.IOException;
import java.io.InputStream;

import net.sf.sfac.gui.editor.ObjectEditor;
import net.sf.sfac.setting.Settings;


/**
 * Processor of a file.
 * 
 * @author berol
 */
public interface FileProcessor {


    public static final int NONE = 0;
    public static final int ONE = 1;
    public static final int MANY = 2;


    /**
     * @return A displayable name for this processor.
     */
    public String getProcessorName();


    /**
     * Initialize this file processor just after instantiation (called only once). <br>
     * The settings passed here should be use to store all persistent data.
     * 
     * @param sett
     *            Settings for this FileProcessor instance.
     */
    public void init(Settings sett);


    /**
     * Get the number of output streams generated for the given number of input streams. <br>
     * possible values are:
     * <ul>
     * <li>NONE
     * <li>ONE
     * <li>MANY
     * </ul>
     * 
     * @param inputCardinality
     *            ONE or MANY.
     */
    public int getOutputCardinality(int inputCardinality);


    /**
     * Notification that the processing of a batch of file is starting. <br>
     * This method allows to initialize this FileProcessor before each file batch processing.
     * 
     * @param env
     */
    public void startBatch(FileProcessorEnvironment env) throws IOException;


    /**
     * Request to process the given input stream. <br>
     * The output stream to use for output should be requested to the FileProcessorEnvironment.
     * 
     * @param is
     *            the InputStream to process
     * @param info
     *            Information about the input file to process
     * @param env
     *            object used by this FileProcessor to interact with its environment.
     * @throws IOException
     *             if something goes wrong.
     */
    public void process(InputStream is, InputFileInfo info, FileProcessorEnvironment env) throws IOException;


    /**
     * Notification that the current batch is finished.
     * 
     * @param env
     */
    public void endBatch(FileProcessorEnvironment env) throws IOException;


    /**
     * Get a GUI editor for this object.
     */
    public ObjectEditor getEditor();


    /**
     * Get the status of this processor to display in the status bar of the GUI during file processing. <br>
     * <p>
     * Warning: this method is called in a different Thread from the processing Thread (the Thread calling the process(..)
     * method). It is usually called from the swing EventDispatchThread. So, be aware of possible threading synchronization
     * problems while accessing the instance variables of your class.
     * </p>
     * 
     * @return the status of this processor.
     */
    public String getStatusMessage();


}
