package net.sf.filePiper.model;


import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import net.sf.sfac.file.FilePathUtils;
import net.sf.sfac.file.InvalidPathException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


public class FileMatcher {


    static Log log = LogFactory.getLog(FileMatcher.class);


    private FilePathUtils baseDir;

    private List<Pattern> includes;
    private List<Pattern> excludes;
    private List<Pattern> directoryIncludes;
    private List<Pattern> directoryExcludes;


    public FileMatcher(File sourceBaseDir, String includesPattern, String excludesPattern) {
        baseDir = new FilePathUtils(sourceBaseDir);
        includesPattern = (includesPattern == null) ? null : includesPattern.replace('\\', '/');
        excludesPattern = (excludesPattern == null) ? null : excludesPattern.replace('\\', '/');
        includes = getProcessedPatterns(includesPattern, true);
        excludes = getProcessedPatterns(excludesPattern, false);
        directoryIncludes = getDirectoryIncludes(includesPattern);
        directoryExcludes = getDirectoryExcludes(excludesPattern);
    }


    public boolean matchFile(File fil) {
        return matches(fil, includes, excludes);
    }


    public boolean matchDirectoryTree(File fil) {
        return matches(fil, directoryIncludes, directoryExcludes);
    }


    private boolean matches(File fil, List<Pattern> incl, List<Pattern> excl) {
        if ((incl == null) && (excl == null)) return true;
        String filePath;
        try {
            filePath = baseDir.getRelativeFilePath(fil.getAbsolutePath());
        } catch (InvalidPathException e) {
            throw new IllegalArgumentException("Inavlid path: " + fil.getAbsolutePath(), e);
        }
        if (filePath.equals(".")) return true;
        boolean included = patternMatches(filePath, incl, true);
        if (included) {
            return !patternMatches(filePath, excl, false);
        }
        return false;
    }


    private List<Pattern> getDirectoryIncludes(String includesAntPattern) {
        List<Pattern> result = null;
        for (String antPattern : tokenizePattern(includesAntPattern)) {
            antPattern = normalize(antPattern);
            if (antPattern.startsWith("**")) {
                // if at least one include pattern start with '**', we cannot build includes at directory level
                result = null;
                break;
            } else {
                int includeEnd = -1;
                int doubleStarIndex = antPattern.indexOf("**");
                boolean expand = false;
                if (doubleStarIndex > 0) {
                    includeEnd = doubleStarIndex - 1;
                    expand = true;
                } else {
                    int lastSlashIndex = antPattern.lastIndexOf("/");
                    if (lastSlashIndex > 0) includeEnd = lastSlashIndex;
                }
                if (includeEnd > 0) {
                    String[] pathElements = antPattern.substring(0, includeEnd).split("/");
                    StringBuffer pathBuffer = new StringBuffer();
                    for (String el : pathElements) {
                        if (pathBuffer.length() > 0) pathBuffer.append("/");
                        pathBuffer.append(el);
                        String path = pathBuffer.toString();
                        Pattern pat = getProcessedPattern(path, false);
                        if (pat != null) {
                            log.info("Directory include pattern = " + path + " = " + pat);
                            if (result == null) result = new ArrayList<Pattern>();
                            result.add(pat);
                        }
                    }
                    if (expand) {
                        String path = pathBuffer.toString();
                        Pattern pat = getProcessedPattern(path, true);
                        if (pat != null) {
                            log.info("Directory include pattern = " + path + "/** = " + pat);
                            if (result == null) result = new ArrayList<Pattern>();
                            result.add(pat);
                        }
                    }
                }
            }
        }
        return result;
    }


    private List<Pattern> getDirectoryExcludes(String excludesAntPattern) {
        List<Pattern> result = null;
        for (String antPattern : tokenizePattern(excludesAntPattern)) {
            antPattern = normalize(antPattern);
            if (antPattern.endsWith("/**/*")) {
                String subPattern = antPattern.substring(0, antPattern.length() - 5);
                Pattern pat = getProcessedPattern(subPattern, false);
                if (pat != null) {
                    log.info("Directory exclude pattern = " + subPattern + " = " + pat);
                    if (result == null) result = new ArrayList<Pattern>();
                    result.add(pat);
                }
            }
        }
        return result;
    }


    private String normalize(String antPattern) {
        if (antPattern.endsWith("**")) return antPattern + "/*";
        if (antPattern.endsWith("/")) return antPattern + "**/*";
        return antPattern;
    }


    private List<Pattern> getProcessedPatterns(String antPatternList, boolean include) {
        List<Pattern> result = null;
        for (String antPattern : tokenizePattern(antPatternList)) {
            antPattern = normalize(antPattern);
            Pattern pat = getProcessedPattern(antPattern, false);
            if (pat != null) {
                log.info("File " + (include ? "include" : "exclude") + " pattern = " + antPattern + " = " + pat);
                if (result == null) result = new ArrayList<Pattern>();
                result.add(pat);

            }
        }
        return result;
    }


    private String[] tokenizePattern(String antPattern) {
        String[] tokens = new String[0];
        if (antPattern != null) {
            antPattern = antPattern.trim();
            if (!antPattern.equals("")) {
                tokens = antPattern.split("\\s*,\\s*");
            }
        }
        return tokens;
    }


    private Pattern getProcessedPattern(String antPattern, boolean expanded) {
        if (antPattern == null) return null;
        antPattern = antPattern.trim();
        if (antPattern.equals("")) return null;
        // build the regular expression corresponding to the given ant-like pattern
        StringBuffer regExpr = new StringBuffer();
        int len = antPattern.length();
        String separatorPattern = (File.separatorChar == '\\') ? "\\\\" : File.separator;
        for (int i = 0; i < len; i++) {
            char ch = antPattern.charAt(i);
            switch (ch) {
                case '[':
                case ']':
                case '(':
                case ')':
                case '.':
                    regExpr.append("\\");
                    regExpr.append(ch);
                    break;
                case '/':
                    regExpr.append(separatorPattern);
                    break;
                case '?':
                    regExpr.append("[^");
                    regExpr.append(separatorPattern);
                    regExpr.append("]");
                    break;
                case '*':
                    if (antPattern.regionMatches(i, "**/", 0, 3)) {
                        regExpr.append("(.*");
                        regExpr.append(separatorPattern);
                        regExpr.append(")*");
                        i += 2;
                    } else {
                        regExpr.append("[^");
                        regExpr.append(separatorPattern);
                        regExpr.append("]*");
                    }
                    break;
                default:
                    regExpr.append(ch);
            }
        }
        if (expanded) regExpr.append(".*");
        return Pattern.compile(regExpr.toString());
    }


    private boolean patternMatches(String filePath, List<Pattern> patterns, boolean defaultMatch) {
        boolean match = defaultMatch;
        if ((patterns != null) && (patterns.size() > 0)) {
            match = false;
            for (Pattern includePattern : patterns) {
                if (includePattern.matcher(filePath).matches()) {
                    match = true;
                    break;
                }
            }
        }
        return match;
    }


}
