/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package net.sourceforge.fastupload.springmvc;

import java.io.File;
import java.io.IOException;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;

import net.sourceforge.fastupload.FastUploadParser;
import net.sourceforge.fastupload.MultiPart;
import net.sourceforge.fastupload.ProgressListener;

import org.springframework.web.context.ServletContextAware;
import org.springframework.web.multipart.MultipartException;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.multipart.MultipartResolver;
import org.springframework.web.multipart.support.DefaultMultipartHttpServletRequest;
import org.springframework.web.util.WebUtils;

/**
 * @author <a href="mailto:link.qian@yahoo.com">Link Qian</a>
 * @author <a href="lkclkc88@sina.com">lkclkc88@sina.com</a>
 * 
 */
public class FastuploadResolver extends FastUploadSupport implements MultipartResolver, ServletContextAware {

    public FastuploadResolver() {

    }

    @Override
    public void cleanupMultipart(MultipartHttpServletRequest request) {
	String repository = fileFactory.getRepository();
	if (repository != null && repository.trim().length() != 0) {
	    for (MultiPart e : multiParts) {
		if (e.isFile())
		    doCleanup(e);
	    }
	}
    }

    private void doCleanup(MultiPart e) {
	File file = new File(e.getName());
	boolean deleted = file.exists() && file.delete();
	if (!deleted && logger.isWarnEnabled()) {
	    logger.warn("temporary file [" + e.getName() + "] delete failure");
	}
    }

    @Override
    public boolean isMultipart(HttpServletRequest request) {
	return "post".equalsIgnoreCase(request.getMethod()) && FastUploadParser.isMultiPart(request);
    }

    @Override
    public MultipartHttpServletRequest resolveMultipart(HttpServletRequest request) throws MultipartException {
	initFileFactory();
	try {
	    fastUploadParser = new FastUploadParser(request, fileFactory);
	    progressListener = new ProgressListener(fastUploadParser);
	    multiParts = fastUploadParser.parseList();
	    return new DefaultMultipartHttpServletRequest(request, getMultiPartFiles(), getParameters(), getParamContentTypes());
	} catch (IOException e) {
	    if (logger.isErrorEnabled()) {
		logger.error(e);
		throw new MultipartException("resolve multipart/form-data ServletRequest error", e);
	    }
	}

	return null;
    }

    @Override
    public void setServletContext(ServletContext servletContext) {
	if (useTempRepository && tempRepository == null) {
	    fileFactory.setRepository(WebUtils.getTempDir(servletContext).getAbsolutePath());
	}
    }

}
