/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package net.sourceforge.fastupload.springmvc;

import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.sourceforge.fastupload.FastUploadParser;
import net.sourceforge.fastupload.FileFactory;
import net.sourceforge.fastupload.MultiPart;
import net.sourceforge.fastupload.ProgressListener;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.util.WebUtils;

/**
 * 
 * @author <a href="mailto:link.qian@yahoo.com">Link Qian</a>
 * @author <a href="lkclkc88@sina.com">lkclkc88@sina.com</a>
 * 
 */
public abstract class FastUploadSupport {

    private static final int DEFAUT_THRESHOLD = 2000000;

    protected final Log logger = LogFactory.getLog(getClass());

    protected FileFactory fileFactory = FileFactory.getInstance();

    protected FastUploadParser fastUploadParser;

    protected List<MultiPart> multiParts;

    protected boolean useTempRepository = false;

    protected String tempRepository;

    private String encoding;

    private int threshold = DEFAUT_THRESHOLD;

    private long maxContentLength = DEFAUT_THRESHOLD;

    private String allowedTypes;

    private String allowedExtensions;
    
    protected ProgressListener progressListener;

    protected MultiValueMap<String, MultipartFile> getMultiPartFiles() {
	MultiValueMap<String, MultipartFile> multiValueMap = new LinkedMultiValueMap<String, MultipartFile>();
	for (MultiPart e : multiParts) {
	    if (e.isFile())
		multiValueMap.add(e.getFieldName(), new MultipartFileWrapper(e));
	}
	return multiValueMap;
    }

    protected Map<String, String[]> getParameters() {
	Map<String, String[]> map = new HashMap<String, String[]>();
	for (MultiPart e : multiParts) {
	    if (!e.isFile())
		try {
		    String[] values = map.get(e.getFieldName());
		    if (values == null)
			map.put(e.getFieldName(), new String[] { e.getString() });
		    else {
			String[] newParam = StringUtils.addStringToArray(values, e.getString());
			map.put(e.getFieldName(), newParam);
		    }
		} catch (UnsupportedEncodingException e1) {
		    if (logger.isDebugEnabled()) {
			logger.debug(e1);
		    }
		}
	}
	return map;
    }

    protected Map<String, String> getParamContentTypes() {
	Map<String, String> map = new HashMap<String, String>();
	for (MultiPart e : multiParts) {
	    map.put(e.getFieldName(), e.getContentType());
	}
	return map;
    }

    protected void initFileFactory() {
	fileFactory.setCharset(encoding == null ? WebUtils.DEFAULT_CHARACTER_ENCODING : encoding);
	fileFactory.setThreshold(threshold);
	fileFactory.setMaxContentLength(maxContentLength);
	if (useTempRepository)
	    fileFactory.setRepository(tempRepository);
	fileFactory.setAllowedExtensions(allowedExtensions);
	fileFactory.setAllowedTypes(allowedTypes);
    }
    
     
    public ProgressListener getProgressListener() {
        return progressListener;
    }

    public int getThreshold() {
	return threshold;
    }

    public void setThreshold(int threshold) {
	this.threshold = threshold;
    }

    public long getMaxContentLength() {
	return maxContentLength;
    }

    public void setMaxContentLength(long maxContentLength) {
	this.maxContentLength = maxContentLength;
    }

    public String getAllowedTypes() {
	return allowedTypes;
    }

    public void setAllowedTypes(String allowedTypes) {
	this.allowedTypes = allowedTypes;
    }

    public String getAllowedExtensions() {
	return allowedExtensions;
    }

    public void setAllowedExtensions(String allowedExtensions) {
	this.allowedExtensions = allowedExtensions;
    }

    /**
     * @return the encoding
     */
    public String getEncoding() {
	return encoding;
    }

    /**
     * @param encoding
     *            the encoding to set
     */
    public void setEncoding(String encoding) {
	this.encoding = encoding;
    }

    public boolean isUseTempRepository() {
	return useTempRepository;
    }

    public void setUseTempRepository(boolean useTempRepository) {
	this.useTempRepository = useTempRepository;
    }

    public String getTempRepository() {
	return tempRepository;
    }

    public void setTempRepository(String tempRepository) {
	this.tempRepository = tempRepository;
    }

}
