//
// Copyright 2001 - 2010, 2012 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.util;

import com.google.common.base.Strings;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import static java.nio.file.StandardWatchEventKinds.ENTRY_CREATE;
import static java.nio.file.StandardWatchEventKinds.ENTRY_DELETE;
import static java.nio.file.StandardWatchEventKinds.ENTRY_MODIFY;
import static java.nio.file.StandardWatchEventKinds.OVERFLOW;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * {@code Properties} extends
 * {@code java.util.Properties} to include:
 * <ul>
 *   <li>
 *     Loading properties from a named file and returning
 *     a {@code net.sf.eBus.util.Properties} object.
 *   </li>
 *   <li>
 *     Storing a {@link net.sf.eBus.util.Properties} object
 *     into a named file.
 *   </li>
 *   <li>
 *     Returning property values as a boolean, int, double and
 *     String array.
 *   </li>
 *   <li>
 *     Setting property values as a boolean, int, double and
 *     String array.
 *   </li>
 *   <li>
 *     Informs registered {@link PropertiesListener}s when the
 *     underlying properties file has changed and been
 *     automatically reloaded.
 *     <p>
 *     Note: automatic properties file reloading is done only if
 *     there is a registered properties listener. If there are
 *     no registered listeners, then Properties does not watch
 *     the underlying file for changes and so does not
 *     automatically reload the properties file if it should
 *     change.
 *   </li>
 * </ul>
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

// Too late to change name now.
@SuppressWarnings({"java:S2176", "java:S2160"})
public final class Properties
    extends java.util.Properties
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     *  This is eBus version 2.1.0.
     */
    private static final long serialVersionUID = 0x050200L;

    /**
     * Append this suffix to the file name.
     */
    private static final String THREAD_SUFFIX = "WatchThread";

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Once a property file is loaded, store the Properties
     * object in this map under its file name.
     */
    private static Map<String, Properties> sPropertiesMap =
        new HashMap<>();

    /**
     * Properties map synchronizer.
     */
    private static Lock sPropertiesMutex = new ReentrantLock();

    /**
     * Logging subsystem interface.
     */
    private static final Logger sLogger =
        LoggerFactory.getLogger(Properties.class);

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * This properties file is associated with this file.
     */
    private final String mFileName;

    /**
     * Inform these listeners of property file changes.
     */
    private final transient List<PropertiesListener> mListeners;

    /**
     * This thread polls the watch key for file updates.
     */
    private transient WatchThread mWatchThread;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an empty property list with no default values.
     * Because there is no associated file, property listening
     * is not allowed.
     */
    public Properties()
    {
        super ();

        mFileName = null;
        mListeners = new LinkedList<>();
        mWatchThread = null;
    } // end of Properties()

    /**
     * Creates an empty property list with default values.
     * Because there is no associated file, property listening
     * is not allowed.
     * @param defaults the default property values.
     */
    public Properties(final java.util.Properties defaults)
    {
        super (defaults);

        mFileName = null;
        mListeners = new LinkedList<>();
        mWatchThread = null;
    } // end of Properties(java.util.Properties)

    // This properties object cannot be directly instantiated
    // by an application but only through the loadProperties()
    // static methods.
    private Properties(final String fileName)
    {
        super ();

        mFileName = fileName;
        mListeners = new LinkedList<>();
        mWatchThread = null;
    } // end of Properties(String)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    //
    // Get methods, no default.
    //

    /**
     * Returns the named property value as a
     * {@code boolean}. If {@code key} is an unknown
     * property or is neither {@code "true"} nor
     * {@code "false"} then returns {@code false}.
     * @param key The property key.
     * @return the named property value as a
     * {@code boolean}. If the value is {@code "true"},
     * then {@code true} is returned, otherwise returns
     * {@code false}.
     * @see #getBooleanProperty(String, boolean)
     * @see #setBooleanProperty(String, boolean)
     */
    public boolean getBooleanProperty(final String key)
    {
        final String value = getProperty(key);
        boolean retval = false;

        if (value != null)
        {
            final Boolean bool = Boolean.valueOf(value.trim());

            retval = bool;
        }

        return (retval);
    } // end of getBooleanProperty(String)

    /**
     * Returns the named property value as an {@code int}.
     * If {@code key} is an unknown property or is not a
     * valid integer, then returns {@code 0}.
     * @param key The property key.
     * @return the named property value as an {@code int}.
     * @see #getIntProperty(String, int)
     * @see #setIntProperty(String, int)
     */
    public int getIntProperty(final String key)
    {
        final String value = getProperty(key);
        int retval = 0;

        if (value != null && value.length() > 0)
        {
            try
            {
                retval = Integer.parseInt(value.trim());
            }
            catch (NumberFormatException numberex)
            {
                // Ignore - return value already set to 0.
            }
        }

        return (retval);
    } // end of getIntProperty(String)

    /**
     * Returns the named property value as a {@code double}.
     * If {@code key} is an unknown property or is not a
     * value double, then returns {@code 0.0}.
     * @param key The property key.
     * @return the named property value as a {@code double}.
     * @exception NumberFormatException
     * if the property value is not a valid double.
     * @see #getDoubleProperty(String, double)
     * @see #setDoubleProperty(String, double)
     */
    public double getDoubleProperty(final String key)
    {
        final String value = getProperty(key);
        double retval = 0.0;

        if (value != null && value.length() > 0)
        {
            try
            {
                retval = Double.parseDouble(value.trim());
            }
            catch (NumberFormatException numberex)
            {
                // Ignore - return value already set to 0.0.
            }
        }

        return (retval);
    } // end of getDoubleProperty(String)

    /**
     * Returns the named property value as a
     * {@code String[]}. {@code ifs} is used as
     * the interfield separator character.
     * If the property value does not exist, then returns
     * an empty array.
     * @param key The property key.
     * @param ifs The interfield separator.
     * @return the named property value as a
     * {@code String[]}. If the property value does not
     * exist, then returns an empty array.
     * @see #setArrayProperty(String, String[], char)
     */
    public String[] getArrayProperty(final String key,
                                     final char ifs)
    {
        final String value = getProperty(key);
        String[] retval;

        if (value == null || value.length() == 0)
        {
            retval = new String[0];
        }
        else
        {
            // Place the ifs in a regular expression [] block
            // just in case the ifs is also a regular expression
            // character.
            retval = value.split(String.format("[%c]", ifs));
        }

        return (retval);
    } // end of getArrayProperty(String, char)

    //
    // Get methods with default.
    //

    /**
     * Returns the named property value as a
     * {@code boolean}. If {@code key} is an unknown
     * property or an invalid boolean string, then returns the
     * default value.
     * @param key The property key.
     * @param defaultValue The default value.
     * @return the named property value as a
     * {@code boolean}. If {@code key} is an unknown
     * property or an invalid boolean string, then returns
     * {@code defaultValue}.
     * @see #getBooleanProperty(String)
     * @see #setBooleanProperty(String, boolean)
     */
    public boolean getBooleanProperty(final String key,
                                      final boolean defaultValue)
    {
        final String value = getProperty(key);
        boolean retval = defaultValue;

        if (!Strings.isNullOrEmpty(value))
        {
            retval = Boolean.parseBoolean(value.trim());
        }

        return (retval);
    } // end of getBooleanProperties(String, boolean)

    /**
     * Returns the named property value as an {@code int}.
     * If either the property does not exist or does exist but
     * is not an integer, then returns the default value.
     * @param key The property key.
     * @param defaultValue The default value.
     * @return the named property value as an {@code int}
     * or {@code defaultValue}.
     * @see #getIntProperty(String)
     * @see #setIntProperty(String, int)
     */
    public int getIntProperty(final String key,
                              final int defaultValue)
    {
        final String value = getProperty(key);
        int retval = defaultValue;

        if (!Strings.isNullOrEmpty(value))
        {
            try
            {
                retval = Integer.parseInt(value.trim());
            }
            catch (NumberFormatException formex)
            {}
        }

        return (retval);
    } // end of getIntProperty(String, int)

    /**
     * Returns the named property value as a {@code double}.
     * If the property value does not exist or is not a valid
     * {@code double}, then returns the default value.
     * @param key The property key.
     * @param defaultValue The default value.
     * @return the named property value as a {@code double}.
     * If the property value does not exist or is not a valid
     * {@code double}, then returns
     * {@code defaultValue}.
     * @see #getDoubleProperty(String)
     * @see #setDoubleProperty(String, double)
     */
    public double getDoubleProperty(final String key,
                                    final double defaultValue)
    {
        final String value = getProperty(key);
        double retval = defaultValue;

        if (!Strings.isNullOrEmpty(value))
        {
            try
            {
                retval = Double.parseDouble(value);
            }
            catch (NumberFormatException formex)
            {}
        }

        return (retval);
    } // end of getDoubleProperties(String, double)

    /**
     * Returns a set of keys in this property list whose key
     * matches the given regular expression pattern {@code p}
     * and the corresponding values are strings.
     * Includes distinct keys in the default property list if a
     * key of the same name is not in the main properties list.
     * Properties whose key or value is not of type
     * {@code String} are omitted.
     * <p>
     * The returned set is not backed by the {@code Properties}
     * object. Changes to {@code this Properties} are not
     * reflected in the set or vice versa.
     * </p>
     * @param p match property keys against this pattern.
     * @return String property keys matching the regular expression
     * pattern.
     * @see #stringPropertyNames()
     * @see #defaults
     */
    public Set<String> stringPropertyNames(final Pattern p)
    {
        final Iterator<String> it;
        Matcher m;
        final Set<String> retval =
            new TreeSet<>(this.stringPropertyNames());

        it = retval.iterator();
        while (it.hasNext())
        {
            m = p.matcher(it.next());

            // If the property does *not* match the pattern,
            // then remove it from the set.
            if (!m.matches())
            {
                it.remove();
            }
        }

        return (retval);
    } // end of stringPropertyNames(Pattern)

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Sets the named property value to the specified boolean.
     * @param key The property key.
     * @param value The boolean value.
     * @see #getBooleanProperty(String)
     * @see #getBooleanProperty(String, boolean)
     */
    public void setBooleanProperty(final String key,
                                   final boolean value)
    {
        setProperty(key, Boolean.toString(value));
    } // end of setBooleanProperty(String, boolean)

    /**
     * Sets the named property value to the specified integer.
     * @param key The property key.
     * @param value The integer value.
     * @see #getIntProperty(String)
     * @see #getIntProperty(String, int)
     */
    public void setIntProperty(final String key, final int value)
    {
        setProperty(key, Integer.toString(value));
    } // end of setIntProperty(String, int)

    /**
     * Sets the named property value to the specified double.
     * @param key The property key.
     * @param value The double value.
     * @see #getDoubleProperty(String)
     * @see #getDoubleProperty(String, double)
     */
    public void setDoubleProperty(final String key,
                                  final double value)
    {
        setProperty(key, Double.toString(value));
    } // end of setDoubleProperty(String, double)

    /**
     * Sets the named property value to the string array.
     * The array is converted into a single string by
     * concatenating the strings together separated by
     * {@code ifs}.
     * @param key The property key.
     * @param value The string array.
     * @param ifs The interfield separator.
     * @see #getArrayProperty
     */
    public void setArrayProperty(final String key,
                                 final String[] value,
                                 final char ifs)
    {
        StringBuilder buffer;
        int bufferLen;
        int i;
        String newValue = "";

        if (value.length > 0)
        {
            // Figure out the buffer size first by adding up
            // the string length.
            for (i = 0, bufferLen = 0; i < value.length; ++i)
            {
                bufferLen += value[i].length();
            }

            // Add in the IFS.
            bufferLen += value.length - 1;

            // Allocate the buffer and start filling it in.
            buffer = new StringBuilder(bufferLen);
            for (i = 0; i < value.length; ++i)
            {
                if (i > 0)
                {
                    buffer.append(ifs);
                }

                buffer.append(value[i]);
            }

            newValue = buffer.toString();
        }

        setProperty(key, newValue);
    } // end of setArrayProperty(String, String[], char)

    //
    // end of Set Methods.
    //-----------------------------------------------------------

    /**
     * Reloads properties from the properties file.
     * @exception IOException
     * if there are errors reading in the properties file.
     * @see #store(String)
     * @see #loadProperties(String)
     * @see #loadProperties(File)
     */
    public void load()
        throws IOException
    {
        if (mFileName == null)
        {
            throw (new IOException("no property file to load"));
        }

        try (final FileInputStream fis =
                 new FileInputStream(mFileName))
        {
            load(fis);
        }
        catch (FileNotFoundException nofilex)
        {
            // Convert this exception to an IOException.
            throw (
                new IOException(mFileName + " does not exist"));
        }
    } // end of load()

    /**
     * Stores properties in properties file using the provided
     * header comment. The header comment is placed at the
     * property file's start.
     * @param header File header comment. May be
     * {@code null}.
     * @exception FileNotFoundException
     * if the properties file could not be created.
     * @exception IOException
     * if there is an error storing the properties into the
     * file.
     * @see #load
     * @see #loadProperties(String)
     * @see #loadProperties(File)
     */
    public void store(final String header)
        throws IOException
    {
        if (mFileName == null)
        {
            throw (new IOException("no property file"));
        }

        try (final FileOutputStream fos =
                 new FileOutputStream(mFileName))
        {
            store(fos, header);
        }
    } // end of store(String)

    /**
     * Stores properties in the named properties file using the
     * provided header comment. The header comment is placed at
     * the property file's start.
     * @param fileName Property file's name.
     * @param header File header comment. May be
     * {@code null}.
     * @exception IllegalArgumentException
     * if {@code fileName} is either {@code null} or an empty
     * string.
     * @exception FileNotFoundException
     * if the properties file could not be created.
     * @exception IOException
     * if there is an error storing the properties into the
     * file.
     * @see #store(String)
     * @see #store(File, String)
     * @see #load
     * @see #loadProperties(String)
     * @see #loadProperties(File)
     */
    public void store(final String fileName,
                      final String header)
        throws IOException
    {
        if (Strings.isNullOrEmpty(fileName))
        {
            throw (
                new IllegalArgumentException(
                    "null or empty fileName"));
        }

        store(new File(fileName), header);
    } // end of store(String, String)

    /**
     * Stores properties in the named properties file using the
     * provided header comment. The header comment is placed at
     * the property file's start.
     * @param file Property file.
     * @param header File header comment. May be
     * {@code null}.
     * @exception IllegalArgumentException
     * if {@code file} is {@code null}.
     * @exception FileNotFoundException
     * if the properties file could not be created.
     * @exception IOException
     * if there is an error storing the properties into the
     * file.
     * @see #store(String)
     * @see #store(String, String)
     * @see #load
     * @see #loadProperties(String)
     * @see #loadProperties(File)
     */
    public void store(final File file,
                      final String header)
        throws IOException
    {
        boolean existsFlag;

        if (file == null)
        {
            throw (new IllegalArgumentException("null file"));
        }

        // Only non-directory, writeable files may be property
        // files.
        existsFlag = file.exists();
        if (existsFlag && file.isDirectory())
        {
            throw (
                new IOException(
                    file.getName() + " is a directory"));
        }

        if (existsFlag && !file.canWrite())
        {
            throw (
                new IOException(
                    file.getName() + " is unwriteable"));
        }

        try (final FileOutputStream fos =
                 new FileOutputStream(file))
        {
            store(fos, header);
        }
    } // end of store(File, String)

    /**
     * Adds a properties listener. If this is the first listener
     * and the watchRate &gt; zero, then starts the watch timer.
     * <p>
     * Note: when the underlying properties file changes, it will
     * be automatically reloaded prior to calling back to the
     * registered {@link PropertiesListener}s.
     * @param listener Add this properties listener.
     * @exception IllegalArgumentException
     * if {@code listener} is {@code null}.
     * @exception IllegalStateException
     * if this is no underlying property file.
     */
    public void addListener(final PropertiesListener listener)
    {
        boolean addFlag = false;
        int listenerSize = 0;

        if (mFileName == null)
        {
            throw (
                new IllegalStateException(
                    "no underlying property file"));
        }

        if (listener == null)
        {
            throw (
                new IllegalArgumentException("null listener"));
        }

        synchronized (mListeners)
        {
            if (!mListeners.contains(listener))
            {
                mListeners.add(listener);
                addFlag = true;
                listenerSize = mListeners.size();
            }
        }

        // If this is the first listener, then
        // start watching the properties file.
        if (addFlag && listenerSize == 1)
        {
            try
            {
                final File f =
                    (new File(mFileName)).getAbsoluteFile();
                final FileSystem fs = FileSystems.getDefault();
                final Path p = fs.getPath(f.getParent());
                final Path fp = fs.getPath(f.getName());
                final WatchService watcher;
                final WatchKey key;

                sLogger.debug(
                    "Watching {} properties file.", mFileName);

                watcher = fs.newWatchService();
                key = p.register(watcher,
                                 ENTRY_CREATE,
                                 ENTRY_MODIFY,
                                 ENTRY_DELETE);
                mWatchThread =
                    new WatchThread(fp,
                                    key,
                                    watcher,
                                    f.getName() + THREAD_SUFFIX,
                                    this);
                mWatchThread.start();
            }
            catch (IOException ioex)
            {
                sLogger.warn("File watch failed.", ioex);
            }
        }
    } // end of addListener(PropertiesListener)

    /**
     * Removes a properties listener. If there are no more
     * listeners and the watch timer is running, then the
     * timer is canceled.
     * <p>
     * Note: when the watch timer is canceled, Properties
     * will no longer determine if the underlying properties file
     * has changed and so will not automatically reload said file
     * if it should change.
     * @param listener Remove this listener.
     * @exception IllegalStateException
     * if this is no underlying property file.
     */
    public void removeListener(final PropertiesListener listener)
    {
        if (mFileName == null)
        {
            throw (
                new IllegalStateException(
                    "no underlying property file"));
        }

        if (listener != null)
        {
            boolean removeFlag = false;
            boolean emptyFlag = false;

            synchronized (mListeners)
            {
                if (mListeners.contains(listener))
                {
                    mListeners.remove(listener);
                    removeFlag = true;
                    emptyFlag = mListeners.isEmpty();
                }
            }

            // If there are no more listeners, then
            // stop watching the properties file.
            if (removeFlag && emptyFlag)
            {
                mWatchThread.cancel();
                mWatchThread = null;

                sLogger.debug(
                    "Stopped watching {} properties file.",
                    mFileName);
            }
        }
    } // end of removeListener(PropertiesListener)

    /**
     * Returns a properties list loaded with the values found
     * in the named properties file. If the file does not exist,
     * an empty properties object is returned. This allows new
     * properties to be created and stored.
     * @param fileName the properties file name.
     * @return A properties list.
     * @exception IllegalArgumentException
     * if {@code fileName} is either {@code null} or an empty
     * string.
     * @exception IOException
     * if {@code fileName} is not a valid properties
     * file.
     * @see #loadProperties(File)
     * @see #load
     * @see #store(String)
     */
    public static Properties loadProperties(final String fileName)
        throws IOException
    {
        Properties retval = null;

        if (Strings.isNullOrEmpty(fileName))
        {
            throw (
                new IllegalArgumentException(
                    "null or empty fileName"));
        }
        else
        {
            retval = loadProperties(new File(fileName));
        }

        return (retval);
    } // end of loadProperties(String)

    /**
     * Returns a properties list loaded with the values found
     * in the properties file. If the file does not exist,
     * an empty properties object is returned. This allows new
     * properties to be created and stored.
     * @param file the properties file object.
     * @return A properties list.
     * @exception IllegalArgumentException
     * if {@code file} is {@code null}.
     * @exception IOException
     * if {@code file} is not a valid properties file.
     * @see #loadProperties(String)
     * @see #load
     * @see #store(String)
     */
    public static Properties loadProperties(final File file)
        throws IOException
    {
        final boolean existsFlag;
        final String fileName;
        Properties retval = null;

        if (file == null)
        {
            throw (new IllegalArgumentException("null file"));
        }

        // Only non-directory, readable files may be property
        // files.
        existsFlag = file.exists();
        if (existsFlag && file.isDirectory())
        {
            throw (
                new IOException(
                    file.getName() + " is a directory"));
        }

        if (existsFlag && !file.canRead())
        {
            throw (
                new IOException(
                    file.getName() + " is unreadable"));
        }

        fileName = file.getPath();

        sPropertiesMutex.lock();
        try
        {
            retval = sPropertiesMap.get(fileName);

            // Check if this properties file has already been
            // loaded.
            if (retval == null)
            {
                sLogger.debug(
                    "Loading {} properties file.", fileName);

                // No. Create a new properties object and
                // load it up.
                retval =
                    createInstance(
                        file, fileName, existsFlag);
            }
        }
        catch (IOException ioex)
        {
            throw (ioex);
        }
        finally
        {
            sPropertiesMutex.unlock();
        }

        return (retval);
    } // end of loadProperties(File)

    /**
     * Does the actual work of creating and filling a properties
     * instance.
     * @param file load the properties from this file.
     * @param fileName the properties file name.
     * @param existsFlag if {@code true} then read in the
     * properties file; otherwise do not.
     * @return the properties instance.
     * @throws IOException
     * if there is an error loading the properties file.
     */
    private static Properties createInstance(
        final File file,
        final String fileName,
        final boolean existsFlag)
        throws IOException
    {
        final Properties retval = new Properties(fileName);

        sPropertiesMap.put(fileName, retval);

        if (existsFlag)
        {
            try (final FileInputStream fis =
                     new FileInputStream(file))
            {
                    retval.load(fis);
            }
            catch (IOException ioex)
            {
                // Remove the properties object from the
                // map and re-throw the exception.
                sPropertiesMap.remove(fileName);
                throw (ioex);
            }
        }

        return (retval);
    } // end of createInstance(File, String, boolean)

    @SuppressWarnings({"java:S3398"})
    private void handleFileEvent(final WatchEvent<?> event)
    {
        sLogger.debug(
            "Properties event: {}.", (event.kind()).name());

        // Whether the properties file was created, modified or
        // deleted, they are all "updates".
        // Reload the properties and inform the listeners.
        try
        {
            final PropertiesEvent propEvent =
                new PropertiesEvent(this);
            List<PropertiesListener> listeners =
                new LinkedList<>();

            load();

            // Copy the listeners list and use that. That way
            // _listeners can be modified without affecting
            // the loop below.
            synchronized (mListeners)
            {
                listeners.addAll(mListeners);
            }

            // Tell the listeners about the changes but only
            // if the modified properties are successfully
            // loaded.
            listeners.stream().
                forEach(
                    listener ->
                    {
                        try
                        {
                            listener.propertiesUpdate(propEvent);
                        }
                        catch (Exception jex)
                        {
                            sLogger.warn(
                                "Properties listener exception.",
                                jex);
                        }
                    });
        }
        catch (IOException ioex)
        {
            // Ignore.
        }
    } // end of handleFileEvent(WatchEvent<?>)

//---------------------------------------------------------------
// Inner classes.
//

    private static final class WatchThread
        extends Thread
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Look for changes to this file only.
         */
        private final Path mPropertiesFile;

        /**
         * Use to key to watch for changes to this directory.
         */
        private final WatchKey mKey;

        /**
         * The key is registered with this watch service.
         */
        private final WatchService mWatcher;

        /**
         * When a watch event occurs, inform this properties
         * instance.
         */
        private final Properties mOwner;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        public WatchThread(final Path propFile,
                           final WatchKey key,
                           final WatchService watcher,
                           final String name,
                           final Properties owner)
        {
            super (name);

            mPropertiesFile = propFile;
            mKey = key;
            mWatcher = watcher;
            mOwner = owner;
        } // end of WatchThread(...)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Thread Method Overrides.
        //

        /**
         * This thread waits for watch events to occur on the
         * registered watch key and reports them to the owning
         * properties instance. This thread terminates when
         * the watch key is no longer registered with the watch
         * service.
         */
        @Override
        public void run()
        {
            boolean runFlag = true;

            // Continue taking watch events until the no longer
            // registered.
            while (runFlag)
            {
                // Wait for the watch key to be signalled.
                // Why?
                // Because pollEvents() does not block.
                try
                {
                    // Don't bother with the returned key -
                    // it will be the one we already have.
                    mWatcher.take();

                    // Get the latest events and pass them to the
                    // properties instance.
                    mKey.pollEvents().stream()
                        .filter(
                            e -> (e.kind()!= OVERFLOW &&
                                  (e.context()).equals(
                                      mPropertiesFile)))
                        // Ignore overflows.
                        .forEach(mOwner::handleFileEvent);
                }
                catch (InterruptedException interrupt)
                {
                    // Ignore.
                }

                // Reset the key before checking for new events.
                // If reset() returns false, then we are no
                // longer registered with the watch service.
                runFlag = mKey.reset();
            }
        } // end of run()

        //
        // end of Thread Method Overrides.
        //-------------------------------------------------------

        /**
         * Cancels the registration with the watch service.
         * This will cause this thread to stop running.
         */
        public void cancel()
        {
            mKey.cancel();
            this.interrupt();
        } // end of cancel()
    } // end of class WatchThread
} // end of class Properties
