//
// Copyright 2024 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.util;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;


/**
 * Delays {@code String} generation until {@link #toString()}
 * method is called. This class is used to support lazy text
 * generation in {@code org.slf4j.Logger} prior to version 2.
 * (Note: slf4j does support lambda in version 2). Consider
 * the following example:
 * <pre><code>import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

private static final Logger sLogger = LoggerFactory.getLogger(MyClass.class);

final byte[] data = new byte[50_000];
// Initialize data byte array.

sLogger.debug("data contents:\n{}", new LazyString(() -> HexDump.dump(data, "  ");</code></pre>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

@ThreadSafe
@SuppressWarnings({"java:S4973", "java:S2129"})
public final class LazyString
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Use this value to mark that lazy string generation has not
     * yet been done. The static string is deliberately
     * constructed so as to support "==" string comparison.
     */
    private static final String NOT_SET = new String("(not set)");

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * A {@code false} value signals that text has not yet been
     * initialized and {@code true} that it has.
     */
    private final AtomicBoolean mIsInitialized;

    /**
     * Place generated text into this reference. Initialized with
     * {@code null} value to signal that text has not yet been
     * generated.
     */
    private final AtomicReference<String> mText;

    /**
     * Used to generate text on demand and only once.
     */
    private final Supplier<String> mInitializer;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new, un-initialized lazy string instance using
     * the given {@code String} supplier to generate the text
     * when needed.
     * @param initializer generates text on demand.
     */
    public LazyString(final Supplier<String> initializer)
    {
        mIsInitialized = new AtomicBoolean();
        mText = new AtomicReference<>(NOT_SET);
        mInitializer = initializer;
    } // end of LazyString(Supplier<>)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    @Override
    public String toString()
    {
        return (this.get());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns initialized text. If text has not been initialized
     * yet then performs the initialization now.
     * @return initialized text.
     */
    public @Nullable String get()
    {
        String retval;

        // Keep spinning here until the text is initialized.
        // This is necessary since the text generation is
        // considered expensive in terms of processor time and
        // memory so it may take a while.
        // Note: reference comparison is acceptable here since
        // we control the initial reference.
        while ((retval = mText.get()) == NOT_SET)
        {
            if (mIsInitialized.compareAndSet(false, true))
            {
                mText.set(mInitializer.get());
            }
            // Else keep spinning until text is initialized.
        }

        return (retval);
    } // end of get()

    //
    // end of Get Methods.
    //-----------------------------------------------------------
} // end of class LazyString
