//
// Copyright 2001 - 2008, 2016 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.util;

import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;

/**
 * An {@code eBus.util.TimerTask} associates a timer
 * task with a {@link TimerTaskListener}. When the timer expires,
 * {@link TimerTaskListener#handleTimeout(TimerEvent)} where
 * {@link TimerEvent#getSource()} returns the value passed to
 * the {@link #TimerTask(TimerTaskListener, Object)} constructor.
 * <p>
 * In eBus v. 4.2.0, {@code TimerTaskListener} is marked as a
 * {@code @FunctionalInterface}. This allows the listener to be
 * defined using a lambda expression.
 * </p>
 * <p>
 * If {@link TimerTaskListener#handleTimeout(TimerEvent)} throws
 * an exception, the exception stack trace is logged at the
 * warning level.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

// Too late to change name now.
@SuppressWarnings({"java:S2176"})
public final class TimerTask
    extends java.util.TimerTask
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Statics.
    //

    private static final Logger sLogger =
        Logger.getLogger(TimerTask.class.getName());

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Callback to this method when the timer expires.
     */
    private TimerTaskListener mCallback;

    /**
     * The associated value. Placed in the {@link TimerEvent}
     * passed to the listener.
     */
    private final Object mValue;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new {@code TimerTask}. This task
     * is sent to the {@link TimerTaskListener} when the
     * timer expires. The associated value is {@code null}.
     * @param listener Inform this listener of the timeout.
     * @exception NullPointerException
     * if {@code listener} is {@code null}.
     */
    public TimerTask(final TimerTaskListener listener)
    {
        this (listener, null);
    } // end of TimerTask(TimerTaskListener)

    /**
     * Creates a new {@code TimerTask}. This task
     * is sent to
     * {@link TimerTaskListener#handleTimeout(TimerEvent)}
     * when the timer expires. The associated value is
     * {@code value}.
     * @param listener Inform this listener of the timeout.
     * @param value The associated value passed to the listener.
     * May be {@code null}.
     * @exception NullPointerException
     * if {@code listener} is {@code null}.
     */
    public TimerTask(final TimerTaskListener listener,
                     final Object value)
    {
        super ();

        mCallback =
            Objects.requireNonNull(listener, "null listener");
        mValue = value;
    } // end of TimerTask(TimerTaskListener, Object)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // java.util.TimerTask Method Overrides.
    //

    /**
     * Cancels this timer task. See
     * {@code java.util.TimerTask.cancel} for a
     * detailed description of cancel.
     * @return {@code true} if this timer task is scheduled to
     * run once and never ran or is scheduled for repeated
     * execution. Returns {@code false} if this task was
     * scheduled to run once and was executed, if this task
     * was never scheduled or if this task is already canceled.
     */
    @Override
    public boolean cancel()
    {
        mCallback = null;

        return (super.cancel());
    } // end of cancel()

    /**
     * Tells the listener that this timer has expired.
     * <p>
     * <b>Note: Do not call this method!</b> This method is part
     * of {@code java.util.TimerTask} and should be called
     * by {@code java.util.Timer} only.
     */
    @Override
    public void run()
    {
        try
        {
            mCallback.handleTimeout(new TimerEvent(this, mValue));
        }
        catch (Exception jex)
        {
            sLogger.log(Level.WARNING,
                        "TimerTaskListener exception",
                        jex);
        }
    } // end of run()

    //
    // end of java.util.TimerTask Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns optional value associated with this timer task.
     * May return {@code null}.
     * @return optional timer task value.
     */
    public @Nullable Object value()
    {
        return (mValue);
    } // end of value()

    //
    // end of Get Methods.
    //-----------------------------------------------------------
} // end of class TimerTask
