//
// Copyright 2008, 2016 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.text;

import com.google.common.base.Objects;

/**
 * Contains the token's type, corresponding Java object, raw
 * text and the input line number where the token was found. This
 * class is immutable.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class Token
{

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Locals.
    //

    private final int mType;
    private final Object mValue;
    private final String mToken;
    private final int mLineNumber;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a token with the given attributes.
     * @param type The token type. See {@link TokenLexer} for the
     * token type list.
     * @param value The token value created from raw token
     * string.
     * @param token The raw token string.
     * @param line The input line where the token was found.
     */
    public Token(final int type,
                 final Object value,
                 final String token,
                 final int line)
    {
        mType = type;
        mValue = value;
        mToken = token;
        mLineNumber = line;
    } // end of Token(int, Object, String, int)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the token type.
     * @return the token type.
     */
    public int type()
    {
        return (mType);
    } // end of type()

    /**
     * Returns the token value.
     * @return the token value.
     */
    public Object value()
    {
        return (mValue);
    } // end of value()

    /**
     * Returns the raw token string.
     * @return the raw token string.
     */
    public String token()
    {
        return (mToken);
    } // end of token()

    /**
     * Returns the input line number where the token was found.
     * @return the input line number where the token was found.
     */
    public int lineNumber()
    {
        return (mLineNumber);
    } // end of lineNumber()

    /**
     * Returns {@code true} if {@code value} equals the
     * token value and {@code false} otherwise.
     * @param value Test equality between this object and the
     * token value.
     * @return {@code true} if {@code value} equals the
     * token value and {@code false} otherwise.
     */
    public boolean isValue(final Object value)
    {
        return (Objects.equal(mValue, value));
    } // end of isValue(Object)

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Returns {@code true} if {@code o} is a non-{@code null}
     * {@code Token} instance of the same type and raw token
     * text; {@code false} otherwise.
     * @param o Test equality with this object.
     * @return {@code true} if {@code o} is a non-{@code null}
     * {@code Token} instance of the same type and raw token
     * text; {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (!retcode && o instanceof Token)
        {
            final Token token = (Token) o;

            retcode =
                (mType == token.mType &&
                 mToken.equals(token.mToken));
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the token hashcode based on the raw token string.
     * @return the token hashcode based on the raw token string.
     */
    @Override
    public int hashCode()
    {
        return (mToken.hashCode());
    } // end of hashCode()

    /**
     * Returns this token's textual representation.
     * @return this token's textual representation.
     */
    @Override
    public String toString()
    {
        return (
            String.format(
                "{line=%,d; type=%d; value=\"%s\"; token=\"%s\"}",
                mLineNumber,
                mType,
                mValue,
                mToken));
    } // end of toString()
} // end of class Token
