//
// Copyright 2019, 2020, 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.util;

import com.typesafe.config.ConfigException.ValidationProblem;
import java.util.List;

/**
 * This exception is thrown when message build fails due to an
 * incorrectly configured message. This will be due to either
 * required fields not being set or two or more fields set to
 * incompatible values. That is setting one field to a given
 * value restricts the values allowed for one or more other
 * fields. This method is "fail slow" meaning that it should
 * contain all the {@link ValidationProblem}s found with the
 * configured message and <em>not</em> the first problem.
 * <p>
 * This exception should not be thrown if a single field is set
 * to an invalid value. Instead an exception is thrown by the
 * setter method when given that invalid value.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class ValidationException
    extends RuntimeException
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Build failed attempting to build this message object
     * class.
     */
    private final Class<?> mClass;

    /**
     * List of invalid field names and message explaining why
     * the field is invalid.
     */
    private final transient List<MultiKey2<String, String>> mProblems;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new instance of {@code ValidationException}
     * for the given class and validation problems.
     * @param tc validation exception applies to building this
     * class.
     * @param problems discovered message class problems.
     */
    public ValidationException(final Class<?> tc,
                               final List<MultiKey2<String, String>> problems)
    {
        super (asMessage(tc, problems));

        mClass = tc;
        mProblems = problems;
    } // end of ValidationException(Class, List<>)

    /**
     * Creates a new instance of {@code ValidationException}
     * for the given class and validation problems.
     * @param tc validation exception applies to building this
     * class.
     * @param problems discovered message class problems.
     * @param cause underlying cause for this validation
     * exception.
     */
    public ValidationException(final Class<?> tc,
                               final List<MultiKey2<String, String>> problems,
                               final Throwable cause)
    {
        super (asMessage(tc, problems), cause);

        mClass = tc;
        mProblems = problems;
    } // end of ValidationException(Class, List<>, Throwable)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns text displaying the message validation problems.
     * @return exception in text format.
     */
    @Override
    public String toString()
    {
        return (getMessage());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns builder's target class experiencing the validation
     * problems.
     * @return builder target class.
     */
    public Class<?> targetClass()
    {
        return (mClass);
    } // end of targetClass()

    /**
     * Returns the validation problems as a read-only list.
     * @return problems list.
     */
    public List<MultiKey2<String, String>> problems()
    {
        return (mProblems);
    } // end of problems()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Returns text displaying the message validation problems.
     * @param tc invalid message class.
     * @param problems invalid issues list.
     * @return exception in text format.
     */
    private static String asMessage(final Class<?> tc,
                                    final List<MultiKey2<String, String>> problems)
    {
        final StringBuilder retval = new StringBuilder();

        retval.append(tc.getCanonicalName())
              .append(" failed to build due to the following problems:");

        problems.forEach(
            p -> retval.append(Validator.NEWLINE)
                       .append(p.key(Validator.NAME_INDEX))
                       .append(": ")
                       .append(p.key(Validator.MESSAGE_INDEX)));

        return (retval.toString());
    } // end of asMessage(Class, List<>)
} // end of ValidationException
