//
// Copyright 2003 - 2010 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.util.regex;

import java.util.Iterator;
import java.util.SortedSet;

/**
 * This regular expression component matches only those
 * characters listed in its set. This pattern element has an
 * associated minimum and maximum range. The component must match
 * at least the minimum count and up to the maximum count
 * (inclusive).
 * <p>
 * This set may also be negated which means that it will match
 * any character <i>except</i> those listed in its set.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class CharacterSet
    extends Component
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     *  This is eBus version 2.1.0.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The character set is stored as a bit set for quick
     * matching.
     */
    private final SortedSet<Character> mSet;

    /**
     * Set to {@code true} if this set is negated.
     */
    private final boolean mNegationFlag;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates character set (either postive or negative) with
     * the minimum and maximum match count.
     * @param set The character bit set.
     * @param flag {@code true} if this is a negated set.
     * @param minSize Must match at least this many times.
     * @param maxSize May match up to this many times.
     * @param index The component's index within the pattern.
     */
    /* package */ CharacterSet(final SortedSet<Character> set,
                               final boolean flag,
                               final int minSize,
                               final int maxSize,
                               final int index)
    {
        super (CHARACTER_SET, minSize, maxSize, index);

        mSet = set;
        mNegationFlag = flag;
    } // end of CharacterSet(Set<>, boolean, int, int, int)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Component Abstract Methods Implementation.
    //

    /**
     * Returns {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     * A character set is less than a specific character if the
     * set's smallest character is less than {@code c}.
     * @param c Test against this character.
     * @return {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean lessThan(final char c)
    {
        return (((mSet.first()) < c) || mNegationFlag);
    } // end of lessThan(char)

    /**
     * Returns {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     * A character set is equal to a specific characer if
     * {@code c} is in the set.
     * @param c Test against this character.
     * @return {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean equalTo(final char c)
    {
        return (mSet.contains(c) != mNegationFlag);
    } // end of equalTo(char)

    /**
     * Returns {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     * A character set is greater than a specified character if
     * the set's maximum character is greater than {@code c}.
     * @param c Test against this character.
     * @return {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean greaterThan(final char c)
    {
        return (((mSet.last()) > c) || mNegationFlag);
    } // end of greaterThan(char)

    //
    // end of Component Abstract Methods Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns {@code true} if this character is negated (^)
     * and {@code false} otherwise.
     * @return {@code true} if this character is negated (^)
     * and {@code false} otherwise.
     */
    public boolean isNegated()
    {
        return (mNegationFlag);
    } // end of isNegated()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Returns a textual representation of a character set.
     * @return a textual representation of a character set.
     */
    @Override
    @SuppressWarnings({"java:S3776"})
    public String toString()
    {
        final Iterator<Character>  it = mSet.iterator();
        int index;
        final int[] range = new int[2];
        int rangeIndex = -1;
        final StringBuilder buffer = new StringBuilder();

        buffer.append('[');

        if (mNegationFlag)
        {
            buffer.append('^');
        }

        while (it.hasNext())
        {
            index = it.next();

            // Is a range being collected?
            if (rangeIndex >= 0)
            {
                // Is this character a part of this range?
                if (index == (range[rangeIndex] + 1))
                {
                    // Yes. Add this character to the range.
                    range[1] = index;
                    rangeIndex = 1;
                }
                // No, this is not a part of a character range.
                else
                {
                    // Was a full range collected?
                    if (rangeIndex == 1)
                    {
                        // Yes. Output the range.
                        appendRange(range, buffer);
                    }
                    // No. Out the single character.
                    else
                    {
                        buffer.append((char) range[0]);
                    }

                    // The range is now empty.
                    rangeIndex = -1;
                }
            }

            // If the potential range is empty, then place this
            // character into the range.
            if (rangeIndex < 0)
            {
                range[0] = index;
                rangeIndex = 0;
            }
        }

        // Output any and all waiting characters.
        // Was a full range collected?
        if (rangeIndex == 1)
        {
            // Yes. Output the range.
            appendRange(range, buffer);
        }
        else
        {
            // No. Output the singleton character.
            buffer.append((char) range[0]);
        }

        buffer.append(']');
        appendSize(buffer);

        return (buffer.toString());
    } // end of toString()

    /**
     * Outputs the character range.
     * @param range character range.
     * @param buffer output range to this buffer.
     */
    private void appendRange(final int[] range,
                             final StringBuilder buffer)
    {
        buffer.append((char) range[0]);

        // No. Output the range. If the range is two consecutive
        //characters, then don't use a dash.
        if (range[1] > (range[0] + 1))
        {
            buffer.append('-');
        }

        buffer.append((char) range[1]);
    } // end of appendRange(int[], StringBuilder)
} // end of class CharacterSet
