//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.util;

import com.google.common.base.Strings;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.function.BiPredicate;
import java.util.function.Predicate;

/**
 * This class is used to collect invalid builder settings,
 * combining the invalid field names with text explaining why
 * the field is invalid. Helper methods are provided to perform
 * the most common checks (not null, not null or empty string,
 * and equals value). If the check fails, then the given field
 * name and appropriate reason are added to the errors list.
 * <p>
 * If validation requires more sophisticated validation,
 * {@link #requireTrue(Predicate, Object, String, String)}
 * and
 * {@link #requireTrue(BiPredicate, Object, Object, String, String, String)}
 * are provided. The user provides a predicate which tests either
 * a single field or two fields in combination. If
 * {@code predicate.test} returns {@code false}, then the
 * field or fields are marked as invalid. The {@code BiPredicate}
 * method may be used to test two fields which are mutually
 * dependent. This means that setting one field to a given value
 * limits the value which may be assigned to the other. If this
 * bi-predicate fails, then both fields are in error.
 * </p>
 * <p>
 * If the provided {@code requires} methods are not
 * sufficient (such as testing the mutual validity of three
 * or more fields), then the user must use custom validation
 * code and, if the validation fails, use
 * {@link #addError(MultiKey2)} to enter the error into the
 * validation error list.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public class Validator
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * OS-specific end-of-line string for quick reference.
     */
    public static final String NEWLINE =
        System.getProperty("line.separator");

    /**
     * Standard reason for a required message field not being
     * set.
     */
    public static final String NOT_SET = "not set";

    /**
     * The field name is stored in index {@value}.
     */
    public static final int NAME_INDEX = 0;

    /**
     * The message explaining why the field is invalid is
     * stored in index {@value}.
     */
    public static final int MESSAGE_INDEX = 1;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Contains the invalid field name and the reason why
     * the field is invalid. If the message is valid, then
     * this list will be empty.
     */
    private final List<MultiKey2<String, String>> mErrors;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new validator instance containing no problems.
     */
    public Validator()
    {
        mErrors = new ArrayList<>();
    } // end of Validator()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns the error list as text which each error output
     * as "&lt;field name&gt;: &lt;message&gt;" on a separate
     * line. If there are no errors, then returns an empty
     * string.
     * @return textual representation of error list.
     */
    @Override
    public String toString()
    {
        String sep = "";
        final StringBuilder retval = new StringBuilder();

        for (MultiKey2<String, String> error : mErrors)
        {
            retval.append(sep)
                  .append(error.key(NAME_INDEX))
                  .append(": ")
                  .append(error.key(MESSAGE_INDEX));

            sep = NEWLINE;
        }

        return (retval.toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns {@code true} if there are no validation
     * errors and {@code false} otherwise.
     * @return {@code true} if there are no validation
     * errors.
     */
    public boolean isEmpty()
    {
        return (mErrors.isEmpty());
    } // end of isEmpty()

    /**
     * Returns error count.
     * @return error count.
     */
    public int size()
    {
        return (mErrors.size());
    } // end of size()

    /**
     * Returns an unmodifiable copy of the field errors list.
     * @return field errors list.
     */
    public List<MultiKey2<String, String>> errors()
    {
        return (mErrors.isEmpty() ?
                Collections.emptyList() :
                Collections.unmodifiableList(mErrors));
    } // end of errors()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Adds the given error to the validator error list. If
     * {@code error} is {@code null}, then it is ignored and
     * nothing is added to the errors list. This is done to
     * allow this method to be used in a {@code Validator}
     * method chain
     * @param error invalid field error. Ignored if
     * {@code null}.
     * @return {@code this Validator} instance.
     */
    public Validator addError(final MultiKey2<String, String> error)
    {
        if (error != null)
        {
            mErrors.add(error);
        }

        return (this);
    } // end of addError(MultiKey2<>)

    //
    // end of Set Methods.
    //-----------------------------------------------------------

    /**
     * Checks if the given value is {@code null} and, if so,
     * adds the appropriate error against the field name.
     * @param value field value.
     * @param fieldName name of field containing value.
     * @return {@code this Validator} instance.
     *
     * @see #requireNotNullOrEmpty(String, String)
     */
    public Validator requireNotNull(final Object value,
                                    final String fieldName)
    {
        if (value == null)
        {
            mErrors.add(
                new MultiKey2<>(fieldName, NOT_SET));
        }

        return (this);
    } // end of requireNotNull(Object, String)

    /**
     * Checks if the given {@code String} value is either
     * {@code null} or empty and, if so, adds the appropriate
     * error against the field name.
     * @param value string field value.
     * @param fieldName name of field containing value.
     * @return {@code this Validator} instance.
     *
     * @see #requireNotNull(Object, String)
     */
    public Validator requireNotNullOrEmpty(final String value,
                                           final String fieldName)
    {
        if (Strings.isNullOrEmpty(value))
        {
            mErrors.add(
                new MultiKey2<>(fieldName, "null or empty"));
        }

        return (this);
    } // end of requireNotNullOrEmpty(String, String)

    /**
     * Check if the field value equals the expected value
     * and, if not, adds the appropriate error against the
     * field name.
     * @param value field value.
     * @param expected expected value.
     * @param fieldName name of field containing value.
     * @return {@code this Validator} instance.
     */
    public Validator requireEquals(final Object value,
                                   final Object expected,
                                   final String fieldName)
    {
        if (!Objects.equals(value, expected))
        {
            mErrors.add(
                new MultiKey2<>(
                    fieldName,
                    "does not equal \"" + expected + "\""));
        }

        return (this);
    } // end of requireEquals(Object, Object, String)

    /**
     * If {@code predicate.test(value)} returns {@code false},
     * then the given field name and message are added to the
     * field errors list. This method allows for more
     * sophisticated test beyond check for a {@code null}
     * value or equality.
     * <p>
     * Please note that {@code predicate.test(value)} should
     * return {@code true} if {@code value} is acceptable.
     * <em>Do not write the predicate test method to check
     * if {@code value} is not valid.</em>
     * </p>
     * @param <V> {@code value} type.
     * @param predicate {@code test} method checks if
     * {@code value} is acceptable.
     * @param value field value.
     * @param fieldName name of field containing value.
     * @param message error message if
     * {@code predicate.test(value)} returns {@code false}.
     * @return {@code this Validator} instance.
     *
     * @see #requireTrue(BiPredicate, Object, Object, String, String, String)
     * @see #requireTrue(boolean, String, String)
     */
    public <V> Validator requireTrue(final Predicate<V> predicate,
                                      final V value,
                                      final String fieldName,
                                      final String message)
    {
        if (!predicate.test(value))
        {
            mErrors.add(new MultiKey2<>(fieldName, message));
        }

        return (this);
    } // end of requireTrue(Predicate, V, String, String)

    /**
     * If {@code predicate.test(value)} returns {@code false},
     * then the given field name and message are added to the
     * field errors list. This method allows for tests which
     * require checking two fields against each other when
     * their individual validity is dependent on the fields
     * being mutually consistent. That is, setting the first
     * field to a given value limits what value may be set to
     * the second field.
     * <p>
     * Please note that {@code predicate.test(value)} should
     * return {@code true} if {@code value} is acceptable.
     * <em>Do not write the predicate test method to check
     * if {@code value} is not valid.</em>
     * </p>
     * @param <V1> {@code value1} type.
     * @param <V2> {@code value2} type.
     * @param predicate {@code test} method checks if
     * {@code value1} and {@code value2} are mutually
     * acceptable.
     * @param value1 first field's value.
     * @param value2 second field's value.
     * @param fieldName1 first field's name.
     * @param fieldName2 second field's name.
     * @param message message applied to both fields if
     * {@code predicate.test(value1, value2)} returns
     * {@code false}.
     * @return {@code this Validator} instance.
     *
     * @see #requireTrue(Predicate, Object, String, String)
     * @see #requireTrue(boolean, String, String)
     */
    public <V1, V2> Validator requireTrue(final BiPredicate<V1, V2> predicate,
                                           final V1 value1,
                                           final V2 value2,
                                           final String fieldName1,
                                           final String fieldName2,
                                           final String message)
    {
        if (!predicate.test(value1, value2))
        {
            mErrors.add(new MultiKey2<>(fieldName1, message));
            mErrors.add(new MultiKey2<>(fieldName2, message));
        }

        return (this);
    } // end of requireTrue(...)

    /**
     * If {@code flag} is {@code false}, then an error is
     * raised for the given field name and message.
     * @param flag set to {@code true} if the field is valid
     * and {@code false} is invalid.
     * @param fieldName validating this field name.
     * @param message message explaining invalidation.
     * @return {@code this Validator} instance.
     *
     * @see #requireTrue(Predicate, Object, String, String)
     * @see #requireTrue(BiPredicate, Object, Object, String, String, String)
     */
    public Validator requireTrue(final boolean flag,
                                 final String fieldName,
                                 final String message)
    {
        if (!flag)
        {
            mErrors.add(new MultiKey2<>(fieldName, message));
        }

        return (this);
    } // end of requireTrue(boolean, String)

    /**
     * Throws a {@link ValidationException} if this validator
     * contains any problems. Otherwise does nothing.
     * @param tc validation exception applies to building this
     * class.
     * @throws ValidationException
     * if this validator has any listed problems.
     */
    public void throwException(final Class<?> tc)
    {
        if (!mErrors.isEmpty())
        {
            throw (
                new ValidationException(tc, mErrors));
        }
    } // end of throwException()
} // end of class Validator
