/*
 * Decompiled with CFR 0.152.
 */
package net.sf.eBus.util;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Date;
import java.util.Formatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

public final class IndexCache {
    public static final long DEFAULT_INITIAL_INDEX = 0L;
    public static final long DEFAULT_FINAL_INDEX = Long.MAX_VALUE;
    public static final long DEFAULT_INCREMENT = 1L;
    public static final boolean DEFAULT_AUTO_RESET = false;
    public static final int MAXIMUM_INDEX_NAME_LENGTH = 20;
    private static final String ACCESS_MODE = "rwd";
    private static final long MINIMUM_FILE_LENGTH = 4L;
    private static final long INDEX_OFFSET = 4L;
    private static final long INDEX_SIZE = 54L;
    private static final byte PADDING_VALUE = 0;
    private static final byte[] NAME_PADDING = new byte[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    private static final String CORRUPT_INDEX_CACHE = "corrupt index cache";
    private static final Map<String, IndexCache> sCacheMap = new HashMap<String, IndexCache>();
    private static final Lock sCacheMutex = new ReentrantLock();
    private final Map<String, Index> mIndices;
    private final String mCacheFileName;
    private final RandomAccessFile mCacheFile;
    private final Date mLastModified;

    private IndexCache(Map<String, Index> indices, String cacheFileName, RandomAccessFile cacheFile, Date lastModified) {
        this.mIndices = indices;
        this.mCacheFileName = cacheFileName;
        this.mCacheFile = cacheFile;
        this.mLastModified = lastModified;
    }

    public String cacheFileName() {
        return this.mCacheFileName;
    }

    public Date lastModified() {
        return new Date(this.mLastModified.getTime());
    }

    public boolean isEmpty() {
        return this.mIndices.isEmpty();
    }

    public int indexCount() {
        return this.mIndices.size();
    }

    public List<String> indexNames() {
        return new ArrayList<String>(this.mIndices.keySet());
    }

    public boolean containsIndex(String indexName) {
        return this.mIndices.containsKey(indexName);
    }

    public Index getIndex(String indexName) {
        return this.mIndices.get(indexName);
    }

    public static boolean isCacheOpen(String cacheName) {
        boolean retcode = false;
        sCacheMutex.lock();
        try {
            retcode = sCacheMap.containsKey(cacheName);
        }
        finally {
            sCacheMutex.unlock();
        }
        return retcode;
    }

    public Index addIndex(String indexName, long initialIndex, long finalIndex, long increment, boolean resetFlag) {
        Index retval = null;
        if (indexName == null) {
            throw new IllegalArgumentException("null indexName");
        }
        if (indexName.length() == 0) {
            throw new IllegalArgumentException("empty indexName");
        }
        if (indexName.length() > 20) {
            throw new IllegalArgumentException("indexName too long");
        }
        if (this.mIndices.containsKey(indexName)) {
            throw new IllegalArgumentException("indexName not unique");
        }
        if (increment == 0L) {
            throw new IllegalArgumentException("zero increment");
        }
        if (increment > 0L && initialIndex > finalIndex) {
            throw new IllegalArgumentException("increment > 0 and initialIndex > finalIndex");
        }
        if (increment < 0L && initialIndex < finalIndex) {
            throw new IllegalArgumentException("increment < 0 and initialIndex < finalIndex");
        }
        retval = new Index(indexName, initialIndex, finalIndex, increment, resetFlag, initialIndex);
        this.mIndices.put(indexName, retval);
        return retval;
    }

    public boolean removeIndex(String indexName) {
        boolean retcode = this.mIndices.containsKey(indexName);
        if (retcode) {
            this.mIndices.remove(indexName);
        }
        return retcode;
    }

    public void resetAllIndices() {
        this.mIndices.values().stream().forEach(Index::resetIndex);
    }

    public void clearIndices() {
        this.mIndices.clear();
    }

    public void close() {
        this.doClose();
        sCacheMutex.lock();
        try {
            sCacheMap.remove(this.mCacheFileName);
        }
        finally {
            sCacheMutex.unlock();
        }
    }

    public String toString() {
        StringBuilder retval = new StringBuilder();
        retval.append("cache file = ").append(this.mCacheFileName).append("\n   indices =");
        if (this.mIndices.isEmpty()) {
            retval.append(" (none)");
        } else {
            this.mIndices.values().stream().forEach(index -> retval.append('\n').append(index));
        }
        return retval.toString();
    }

    public static IndexCache open(String cacheFileName) throws IOException {
        IndexCache retval = null;
        if (cacheFileName == null) {
            throw new IllegalArgumentException("null cacheFileName");
        }
        if (cacheFileName.length() == 0) {
            throw new IllegalArgumentException("empty cacheFileName");
        }
        sCacheMutex.lock();
        try {
            File fInfo = new File(cacheFileName);
            if (sCacheMap.containsKey(cacheFileName)) {
                retval = sCacheMap.get(cacheFileName);
            } else if (!fInfo.exists() || fInfo.length() == 0L) {
                retval = IndexCache.newCache(cacheFileName);
                sCacheMap.put(cacheFileName, retval);
            } else {
                if (fInfo.length() < 4L) {
                    throw new IOException(String.format("%s invalid, %,d bytes", cacheFileName, 4L));
                }
                retval = IndexCache.openCache(cacheFileName, fInfo);
                sCacheMap.put(cacheFileName, retval);
            }
        }
        finally {
            sCacheMutex.unlock();
        }
        return retval;
    }

    private static IndexCache newCache(String fn) throws IOException {
        RandomAccessFile fs = new RandomAccessFile(fn, ACCESS_MODE);
        HashMap<String, Index> indices = new HashMap<String, Index>();
        return new IndexCache(indices, fn, fs, new Date());
    }

    private static IndexCache openCache(String fn, File fInfo) throws IOException {
        IndexCache retval;
        try (RandomAccessFile fs = new RandomAccessFile(fn, ACCESS_MODE);){
            HashMap<String, Index> indices = new HashMap<String, Index>();
            int indexCount = fs.readInt();
            long fileSize = 4L + (long)indexCount * 54L;
            if (indexCount < 0 || fileSize > fInfo.length()) {
                throw new IOException(CORRUPT_INDEX_CACHE);
            }
            for (int i = 0; i < indexCount; ++i) {
                Index index = IndexCache.readIndex(fs);
                indices.put(index.name(), index);
            }
            retval = new IndexCache(indices, fn, fs, new Date(fInfo.lastModified()));
        }
        return retval;
    }

    private static Index readIndex(RandomAccessFile fs) throws IOException {
        String indexName = IndexCache.readName(fs);
        long initialIndex = fs.readLong();
        long finalIndex = fs.readLong();
        long increment = fs.readLong();
        boolean resetFlag = fs.readBoolean();
        long currentIndex = fs.readLong();
        if (increment == 0L || increment > 0L && initialIndex > finalIndex || increment < 0L && initialIndex < finalIndex || increment > 0L && currentIndex > finalIndex || increment < 0L && currentIndex < finalIndex) {
            throw new IOException(CORRUPT_INDEX_CACHE);
        }
        return new Index(indexName, initialIndex, finalIndex, increment, resetFlag, currentIndex);
    }

    private void doClose() {
        try {
            this.mCacheFile.seek(0L);
            this.mCacheFile.writeInt(this.mIndices.size());
            for (Index index : this.mIndices.values()) {
                index.store(this.mCacheFile);
            }
            this.mIndices.clear();
        }
        catch (IOException iOException) {
        }
        finally {
            try {
                this.mCacheFile.close();
            }
            catch (IOException iOException) {}
        }
    }

    private static String readName(RandomAccessFile fs) throws IOException {
        int length = fs.read();
        String retval = null;
        if (length <= 0 || length > 20) {
            throw new IOException(CORRUPT_INDEX_CACHE);
        }
        byte[] data = new byte[20];
        if (fs.read(data) < 0) {
            throw new IOException("error reading index name");
        }
        retval = new String(data, StandardCharsets.US_ASCII);
        return retval;
    }

    private static void writeName(String name, RandomAccessFile fs) throws IOException {
        int length = name.length();
        int paddingLength = 20 - length;
        fs.write(length);
        fs.write(name.getBytes(StandardCharsets.US_ASCII));
        if (paddingLength > 0) {
            fs.write(NAME_PADDING, 0, paddingLength);
        }
    }

    static {
        Runtime.getRuntime().addShutdownHook(new CacheShutdown());
    }

    private static final class CacheShutdown
    extends Thread {
        private CacheShutdown() {
        }

        @Override
        public void run() {
            ArrayList<IndexCache> caches = new ArrayList<IndexCache>();
            sCacheMutex.lock();
            try {
                caches.addAll(sCacheMap.values());
                sCacheMap.clear();
            }
            finally {
                sCacheMutex.unlock();
            }
            caches.stream().forEach(rec$ -> ((IndexCache)rec$).doClose());
        }
    }

    public static final class Index {
        private final String mName;
        private final long mInitialIndex;
        private final long mFinalIndex;
        private final long mIncrement;
        private final boolean mResetUponFinalFlag;
        private long mCurrentIndex;

        private Index(String name, long initialIndex, long finalIndex, long increment, boolean resetFlag, long currentIndex) {
            this.mName = name;
            this.mInitialIndex = initialIndex;
            this.mFinalIndex = finalIndex;
            this.mIncrement = increment;
            this.mResetUponFinalFlag = resetFlag;
            this.mCurrentIndex = currentIndex;
        }

        public String name() {
            return this.mName;
        }

        public long initialIndex() {
            return this.mInitialIndex;
        }

        public long increment() {
            return this.mIncrement;
        }

        public long finalIndex() {
            return this.mFinalIndex;
        }

        public long currentIndex() {
            return this.mCurrentIndex;
        }

        public boolean doesResetUponFinalIndex() {
            return this.mResetUponFinalFlag;
        }

        public long nextIndex() {
            long retval = this.mCurrentIndex;
            this.mCurrentIndex += this.mIncrement;
            if (retval > this.mFinalIndex && this.mIncrement > 0L || retval < this.mFinalIndex && this.mIncrement < 0L) {
                if (this.mResetUponFinalFlag) {
                    retval = this.mInitialIndex;
                } else {
                    throw new IndexOutOfBoundsException(String.format("cannot increment beyond final index (%,d)", this.mFinalIndex));
                }
            }
            this.mCurrentIndex = retval + this.mIncrement;
            return retval;
        }

        public void currentIndex(long index) {
            if (this.mIncrement > 0L && (index < this.mInitialIndex || index > this.mFinalIndex) || this.mIncrement < 0L && (index > this.mInitialIndex || index < this.mFinalIndex)) {
                throw new IllegalArgumentException("index outside initial, final interval");
            }
            this.mCurrentIndex = index;
        }

        public long resetIndex() {
            this.mCurrentIndex = this.mInitialIndex;
            return this.mCurrentIndex;
        }

        public String toString() {
            String retval;
            try (Formatter output = new Formatter();){
                output.format("[%s]%n", this.mName).format("   initial index = %,d%n", this.mInitialIndex).format("     final index = %,d%n", this.mFinalIndex).format("       increment = %,d%n", this.mIncrement).format("   current index = %,d%n", this.mCurrentIndex).format("reset upon final = %b", this.mResetUponFinalFlag);
                retval = output.toString();
            }
            return retval;
        }

        private void store(RandomAccessFile fs) throws IOException {
            IndexCache.writeName(this.mName, fs);
            fs.writeLong(this.mInitialIndex);
            fs.writeLong(this.mFinalIndex);
            fs.writeLong(this.mIncrement);
            fs.writeBoolean(this.mResetUponFinalFlag);
            fs.writeLong(this.mCurrentIndex);
        }
    }
}

