//
// Copyright 2003 - 2008 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.util.regex;

/**
 * This regular expression component matches one character. This
 * pattern element has an associated minimum and maximum range.
 * The component must match at least the minimum count and up
 * to the maximum count (inclusive).
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class Literal
    extends Component
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     *  This is eBus version 2.1.0.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The literal component's character.
     */
    private final char mC;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a component for the given literal character,
     * minimum and maximum match sizes.
     * @param c The literal character.
     * @param minSize Must match at least this many times.
     * @param maxSize May match up to this many times.
     * @param index The component's index within the pattern.
     */
    /* package */ Literal(final char c,
                          final int minSize,
                          final int maxSize,
                          final int index)
    {
        super (LITERAL, minSize, maxSize, index);

        mC = c;
    } // end of Literal(char, int, int, int)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Component Abstract Methods Implementation.
    //

    /**
     * Returns {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean lessThan(final char c)
    {
        return (mC < c);
    } // end of lessThan(char)

    /**
     * Returns {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean equalTo(final char c)
    {
        return (mC == c);
    } // end of equalTo(char)

    /**
     * Returns {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean greaterThan(final char c)
    {
        return (mC > c);
    } // end of greaterThan(char)

    //
    // end of Component Abstract Methods Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the literal character.
     * @return the literal character.
     */
    public int literal()
    {
        return (mC);
    } // end of literal()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Returns a textual representation of a literal.
     * @return a textual representation of a literal.
     */
    @Override
    public String toString()
    {
        final StringBuilder buffer = new StringBuilder();

        buffer.append(mC);
        appendSize(buffer);

        return (buffer.toString());
    } // end of toString()
} // end of class Literal
