//
// Copyright 2003 - 2008 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.util.regex;

import java.io.Serializable;

/**
 * Abstract base class for all regular expression components.
 * Every component has a minimum and maximum match length. The
 * component must match at least the minimum length and at most
 * the maximum length (inclusive).
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public abstract class Component
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * The maximum range limit is set to this value when the
     * the limit is infinite.
     */
    public static final int NO_MAX_MATCH_LIMIT = -1;

    // The valid component types.

    /**
     * A literal component matches exactly.
     */
    public static final int LITERAL       = 1;

    /**
     * A character set component matches only those characters
     * in the set (positive) or not in the set (negative).
     */
    public static final int CHARACTER_SET = 2;

    /**
     * This component matches any character.
     */
    public static final int ANY_CHAR      = 3;

    /**
     * The serialization version.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The component's type. Must be one of
     * {@code LITERAL}, {@code CHARACTER_SET} or
     * {@code ANY_CHAR}.
     */
    protected final int mType;

    /**
     * The component's minimum possible length.
     */
    protected final int mMinimumSize;

    /**
     * The component's maximum possible length.
     */
    protected final int mMaximumSize;

    /**
     * The component's position within the regular expression
     * pattern. Indexing begins at zero.
     */
    protected final int mIndex;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Sets the component's type, minimum and maximum length.
     * @param type The regular expression component type.
     * @param minLength The component's minimum length.
     * @param maxLength The component's maximum length.
     * @param index The component's position in the pattern.
     */
    protected Component(final int type,
                        final int minLength,
                        final int maxLength,
                        final int index)
    {
        mType = type;
        mMinimumSize = minLength;
        mMaximumSize = maxLength;
        mIndex = index;
    } // end of Component(int, int, int, int)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns this component's type. Return value will be one of
     * <ul>
     *   <li>
     *     {@code LITERAL}
     *   </li>
     *   <li>
     *     {@code CHARACTER_SET}
     *   </li>
     *   <li>
     *     {@code ANY_CHAR}
     *   </li>
     * </ul>
     * @return this component's type.
     */
    public final int type()
    {
        return (mType);
    } // end of type()

    /**
     * Returns the component's minimum size. This component must
     * match at least this many times.
     * @return the component's minimum size.
     */
    public final int minimumSize()
    {
        return (mMinimumSize);
    } // end of minimumSize()

    /**
     * Returns the component's maximum size. This component must
     * match at most this many times (inclusive).
     * @return the component's maximum size.
     */
    public final int maximumSize()
    {
        return (mMaximumSize);
    } // end of maximumSize()

    /**
     * Returns the component's pattern index.
     * @return the component's pattern index.
     */
    public final int index()
    {
        return (mIndex);
    } // end of index()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Declarations.
    //

    /**
     * Returns {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     */
    public abstract boolean lessThan(char c);

    /**
     * Returns {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     */
    public abstract boolean equalTo(char c);

    /**
     * Returns {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     */
    public abstract boolean greaterThan(char c);

    //
    // Abstract Method Declarations.
    //-----------------------------------------------------------

    /**
     * Adds the size to the output only if not {1, 1}.
     * @param buffer Append the size to this output buffer.
     */
    protected void appendSize(final StringBuilder buffer)
    {
        // Is there a size to output?
        if (mMinimumSize == 1 && mMaximumSize == 1)
        {
            // No. Do nothing.
        }
        // * is {0,} (Kleene closure).
        else if (mMinimumSize == 0 &&
                 mMaximumSize == NO_MAX_MATCH_LIMIT)
        {
            buffer.append('*');
        }
        // + is {1,} (Transitive closure).
        else if (mMinimumSize == 1 &&
                 mMaximumSize == NO_MAX_MATCH_LIMIT)
        {
            buffer.append('+');
        }
        else
        {
            buffer.append('{');
            if (mMinimumSize > 0)
            {
                buffer.append(mMinimumSize);
            }
            buffer.append(',');
            if (mMaximumSize > 0)
            {
                buffer.append(mMaximumSize);
            }
            buffer.append('}');
        }
    } // end of appendSize(StringBuilder)
} // end of class Component
